<?php
// api/get_lessons.php
header('Content-Type: application/json; charset=utf-8');

require_once '../includes/db.php';
require_once '../includes/auth.php';

if (!isStudent()) {
    http_response_code(403);
    echo json_encode(['error' => 'غير مصرح بالدخول']);
    exit();
}

$student = $_SESSION['user'];
$student_id = $student['id'];

try {
    // جلب جميع class_id المرتبطة بهذا الطالب
    $stmt = $pdo->prepare("
        SELECT c.id, c.name 
        FROM classes c 
        JOIN student_classes sc ON c.id = sc.class_id 
        WHERE sc.student_id = ?
    ");
    $stmt->execute([$student_id]);
    $classes = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($classes)) {
        echo json_encode(['lessons' => [], 'count' => 0]);
        exit();
    }

    $class_ids = array_column($classes, 'id');
    $placeholders = str_repeat('?,', count($class_ids) - 1) . '?';

    // ✅ التعديل هنا: استخدام FIND_IN_SET لدعم حقل class_ids
    $stmt = $pdo->prepare("
        SELECT DISTINCT l.*, c.name as class_name 
        FROM lessons l 
        JOIN classes c ON FIND_IN_SET(c.id, l.class_ids) > 0 
        WHERE c.id IN ($placeholders) 
        ORDER BY l.start_time ASC, l.created_at DESC
    ");
    $stmt->execute($class_ids);
    $lessons = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $now = new DateTime();

    $data = [
        'lessons' => array_map(function ($lesson) use ($now) {
            $start_time = $lesson['start_time'] ? new DateTime($lesson['start_time']) : null;
            $is_future = $start_time && $now < $start_time;

            return [
                'id' => $lesson['id'],
                'title' => htmlspecialchars($lesson['title']),
                'description' => htmlspecialchars($lesson['description'] ?: 'لا يوجد وصف.'),
                'lesson_type' => $lesson['lesson_type'],
                'zoom_link' => $lesson['zoom_link'] ?? '#',
                'recorded_link' => $lesson['recorded_link'] ?? '#',
                'start_time' => $lesson['start_time'],
                'is_future' => $is_future,
                'class_name' => htmlspecialchars($lesson['class_name'] ?? 'غير معروف'),
                'teacher_name' => htmlspecialchars($lesson['teacher_name'] ?? 'غير معروف')
            ];
        }, $lessons),
        'count' => count($lessons)
    ];

    echo json_encode($data);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'خطأ داخلي في الخادم: ' . $e->getMessage()]);
}
?>