<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isStudent()) {
    header("Location: ../index.php");
    exit();
}

$student = $_SESSION['user'];
$student_id = $student['id'];
$error = '';
$success = '';

// --- جلب الصفوف المرتبطة بالطالب ---
$stmt = $pdo->prepare("
    SELECT c.id, c.name 
    FROM classes c 
    JOIN student_classes sc ON c.id = sc.class_id 
    WHERE sc.student_id = ?
    ORDER BY c.name
");
$stmt->execute([$student_id]);
$classes = $stmt->fetchAll();

if (empty($classes)) {
    die("<h3 style='text-align: center; color: red; margin-top: 50px;'>عذرًا، لم يتم تعيينك في أي صف بعد.</h3>");
}

$class_ids = array_column($classes, 'id');
$placeholders = str_repeat('?,', count($class_ids) - 1) . '?';

// --- جلب الواجبات المطلوبة من الصفوف التي ينتمي إليها الطالب ---
$stmt = $pdo->prepare("
    SELECT h.*, c.name as class_name 
    FROM homeworks h 
    JOIN classes c ON h.class_id = c.id 
    WHERE h.class_id IN ($placeholders) 
      AND h.due_date >= NOW() 
    ORDER BY h.due_date ASC, h.created_at DESC
");
$stmt->execute($class_ids);
$homeworks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// تحويل المصفوفة لتصبح مفتاحها هو id الواجب (لتسهيل التحقق لاحقًا)
$homeworks_map = [];
foreach ($homeworks as $hw) {
    $homeworks_map[$hw['id']] = $hw;
}

// --- جلب الواجبات التي سبق للطالب تسليمها ---
$stmt = $pdo->prepare("
    SELECT s.homework_id 
    FROM submissions s 
    WHERE s.student_id = ?
");
$stmt->execute([$student_id]);
$submitted_homeworks = $stmt->fetchAll(PDO::FETCH_COLUMN);

// تحويل إلى مصفوفة لسهولة التحقق
$submitted_ids = array_flip($submitted_homeworks);

// --- معالجة تسليم الواجب ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['submit_homework'])) {
    $homework_id = $_POST['homework_id'];
    $note = trim($_POST['note']);

    // التحقق من أن الواجب موجود ومرتبط بصف الطالب
    if (!isset($homeworks_map[$homework_id])) {
        $error = "الواجب غير صالح.";
    } else {
        $hw = $homeworks_map[$homework_id];
        // تحقق إضافي للأمان: هل هذا الواجب من أحد صفوف الطالب؟
        if (!in_array($hw['class_id'], $class_ids)) {
            $error = "الواجب غير صالح.";
        } elseif (isset($submitted_ids[$homework_id])) {
            $error = "لقد سلمت هذا الواجب مسبقًا.";
        } elseif (empty($_FILES['file']['name'])) {
            $error = "يرجى رفع ملف الواجب.";
        } else {
            $ext = pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION);
            $allowed = ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'];
            if (!in_array(strtolower($ext), $allowed)) {
                $error = "نوع الملف غير مسموح به. فقط PDF، صور، أو Word.";
            } else {
                $filename = 'submissions/' . uniqid() . '.' . $ext;
                $path = '../assets/uploads/' . $filename;

                if (move_uploaded_file($_FILES['file']['tmp_name'], $path)) {
                    try {
                        $stmt = $pdo->prepare("INSERT INTO submissions (homework_id, student_id, file_path, note) VALUES (?, ?, ?, ?)");
                        $stmt->execute([$homework_id, $student_id, $filename, $note]);
                        $success = "تم تسليم الواجب بنجاح!";
                    } catch (Exception $e) {
                        $error = "حدث خطأ أثناء التسليم.";
                    }
                } else {
                    $error = "فشل رفع الملف.";
                }
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>الواجبات - اشراق</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        /* شريط التنقل العلوي */
        .top-nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding: 15px 25px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-weight: 700;
            font-size: 20px;
            color: var(--primary);
        }

        .nav-links {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        .nav-link {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 15px;
            border-radius: var(--border-radius);
            color: var(--dark);
            text-decoration: none;
            transition: var(--transition);
        }

        .nav-link:hover {
            background: var(--light);
            color: var(--primary);
        }

        .nav-link.active {
            background: var(--primary);
            color: white;
        }

        /* رأس الصفحة */
        .page-header {
            text-align: center;
            margin-bottom: 30px;
        }

        .page-header h1 {
            font-size: 32px;
            color: var(--primary);
            margin-bottom: 10px;
        }

        .page-header p {
            color: var(--gray);
            font-size: 18px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* قائمة الواجبات */
        .homework-section {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
        }

        .section-title {
            font-size: 22px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .refresh-btn {
            background: #f8f9fa;
            border: 1px solid #ddd;
            padding: 10px 18px;
            border-radius: var(--border-radius);
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 5px;
            transition: var(--transition);
        }

        .refresh-btn:hover {
            background: #e9ecef;
        }

        .homework-card {
            background: #fafafa;
            border-left: 4px solid var(--primary);
            padding: 20px;
            margin-bottom: 20px;
            border-radius: var(--border-radius);
            position: relative;
            overflow: hidden;
        }

        .homework-card::before {
            content: "";
            position: absolute;
            top: 0;
            right: 0;
            width: 8px;
            height: 100%;
            background: linear-gradient(to bottom, var(--primary), var(--secondary));
            opacity: 0.1;
        }

        .homework-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }

        .homework-title {
            font-size: 18px;
            color: var(--dark);
            margin: 0;
        }

        .homework-class {
            background: var(--primary);
            color: white;
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        .homework-due {
            color: var(--error);
            font-weight: 600;
            margin-top: 5px;
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .homework-description {
            color: var(--gray);
            margin-bottom: 15px;
            line-height: 1.6;
        }

        .homework-file {
            margin-bottom: 15px;
            padding: 10px;
            background: #f0f0f0;
            border-radius: 8px;
            text-align: center;
        }

        .homework-file a {
            color: var(--primary);
            text-decoration: underline;
            font-weight: 600;
        }

        .submit-form {
            margin-top: 15px;
            padding: 15px;
            background: #f8f9fa;
            border-radius: var(--border-radius);
            border: 1px dashed #ddd;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }

        .file-input-container {
            position: relative;
            overflow: hidden;
            display: inline-block;
            width: 100%;
        }

        .file-input-button {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
            padding: 12px;
            background: #f8f9fa;
            border: 2px dashed #ddd;
            border-radius: var(--border-radius);
            text-align: center;
            cursor: pointer;
            transition: var(--transition);
            width: 100%;
        }

        .file-input-button:hover {
            background: #e9ecef;
            border-color: var(--primary);
        }

        .file-input-container input[type="file"] {
            position: absolute;
            left: 0;
            top: 0;
            opacity: 0;
            width: 100%;
            height: 100%;
            cursor: pointer;
        }

        .form-note {
            font-size: 14px;
            color: var(--gray);
            margin-top: 5px;
        }

        .btn-submit {
            padding: 10px 20px;
            background: var(--primary);
            color: white;
            border: none;
            border-radius: var(--border-radius);
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-submit:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: inline-block;
        }

        .status-submitted {
            background: rgba(76, 201, 240, 0.1);
            color: var(--success);
        }

        .status-pending {
            background: rgba(255, 235, 59, 0.1);
            color: #ff9800;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: var(--gray);
            grid-column: 1 / -1;
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .homework-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
            
            .homework-class,
            .homework-due {
                margin-top: 5px;
            }
            
            .submit-form {
                padding: 15px 10px;
            }
            
            .form-actions {
                flex-direction: column;
                gap: 10px;
            }
            
            .btn-submit {
                width: 100%;
            }
        }

        /* الرسوم المتحركة */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .homework-card {
            animation: fadeIn 0.5s ease;
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- شريط التنقل العلوي -->
        <div class="top-nav">
            <div class="logo">
                <i class="fas fa-graduation-cap"></i>
                <span>اشراق</span>
            </div>
            
            <div class="nav-links">
                <a href="dashboard.php" class="nav-link">
                    <i class="fas fa-home"></i>
                    الرئيسية
                </a>
                <a href="lessons.php" class="nav-link">
                    <i class="fas fa-book-open"></i>
                    الدروس
                </a>
                <a href="assignments.php" class="nav-link active">
                    <i class="fas fa-tasks"></i>
                    الواجبات
                </a>
                <a href="profile.php" class="nav-link">
                    <i class="fas fa-user"></i>
                    الملف الشخصي
                </a>
            </div>
        </div>

        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1><i class="fas fa-tasks"></i> الواجبات</h1>
            <p>عرض وتسليم الواجبات المطلوبة منك</p>
        </div>

        <!-- رسالة الحالة -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <!-- قائمة الواجبات المطلوبة -->
        <div class="homework-section">
            <div class="section-header">
                <h3 class="section-title">
                    <i class="fas fa-list"></i>
                    الواجبات المطلوبة
                </h3>
            </div>

            <?php if (count($homeworks) > 0): ?>
                <?php foreach ($homeworks as $hw): ?>
                    <div class="homework-card">
                        <div class="homework-header">
                            <h2 class="homework-title"><?php echo htmlspecialchars($hw['title']); ?></h2>
                            <span class="homework-class"><?php echo htmlspecialchars($hw['class_name']); ?></span>
                        </div>
                        
                        <p class="homework-description"><?php echo htmlspecialchars($hw['description'] ?: 'لا يوجد وصف.'); ?></p>
                        
                        <?php if ($hw['file_path']): ?>
                            <div class="homework-file">
                                <a href="../assets/uploads/<?php echo htmlspecialchars($hw['file_path']); ?>" target="_blank">
                                    <i class="fas fa-file-download"></i> تنزيل الملف المرفق
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <div class="homework-due">
                            <i class="fas fa-clock"></i>
                            تاريخ التسليم: <?php echo date('Y-m-d H:i', strtotime($hw['due_date'])); ?>
                        </div>

                        <?php if (isset($submitted_ids[$hw['id']])): ?>
                            <div style="margin-top: 15px; padding: 10px; background: #e8f5e9; border-radius: 8px; text-align: center;">
                                <span class="status-badge status-submitted">
                                    <i class="fas fa-check"></i> تم التسليم
                                </span>
                            </div>
                        <?php else: ?>
                            <form method="POST" enctype="multipart/form-data" class="submit-form">
                                <input type="hidden" name="submit_homework" value="1">
                                <input type="hidden" name="homework_id" value="<?php echo $hw['id']; ?>">
                                
                                <div class="form-group">
                                    <label class="form-label">رفع الملف *</label>
                                    <div class="file-input-container">
                                        <div class="file-input-button">
                                            <i class="fas fa-cloud-upload-alt"></i>
                                            <span>اختر ملفًا (PDF، صورة، Word)</span>
                                        </div>
                                        <input type="file" name="file" accept=".pdf,.jpg,.jpeg,.png,.doc,.docx" required>
                                    </div>
                                    <p class="form-note">الملفات المسموحة: PDF، JPG، PNG، DOC، DOCX</p>
                                </div>

                                <div class="form-group">
                                    <label class="form-label">ملاحظة (اختياري)</label>
                                    <textarea name="note" rows="3" class="form-input"
                                              placeholder="أضف ملاحظاتك..."></textarea>
                                </div>

                                <button type="submit" class="btn-submit">
                                    <i class="fas fa-paper-plane"></i> تسليم الواجب
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-clipboard-check"></i>
                    <p>لا توجد واجبات مطلوبة حاليًا</p>
                    <p>سيظهر هنا كل واجب يُنشر لك من قبل مدرسك</p>
                </div>
            <?php endif; ?>
        </div>

        <!-- عرض الواجبات المُسلَّمة سابقًا -->
        <?php if (!empty($submitted_ids)): ?>
            <div class="homework-section">
                <div class="section-header">
                    <h3 class="section-title">
                        <i class="fas fa-check-circle"></i>
                        الواجبات المُسلَّمة
                    </h3>
                </div>
                
                <div style="background: white; padding: 20px; border-radius: var(--border-radius); box-shadow: var(--shadow);">
                    <?php
                    $stmt = $pdo->prepare("
                        SELECT s.*, h.title as homework_title, c.name as class_name, h.due_date
                        FROM submissions s
                        JOIN homeworks h ON s.homework_id = h.id
                        JOIN classes c ON h.class_id = c.id
                        WHERE s.student_id = ?
                        ORDER BY s.submitted_at DESC
                    ");
                    $stmt->execute([$student_id]);
                    $submitted = $stmt->fetchAll();
                    ?>
                    
                    <?php if (count($submitted) > 0): ?>
                        <table style="width: 100%; border-collapse: collapse; margin-top: 15px; direction: rtl;">
                            <thead>
                                <tr style="background: #007cba; color: white;">
                                    <th style="padding: 12px; text-align: right;">اسم الواجب</th>
                                    <th style="padding: 12px; text-align: right;">الصف</th>
                                    <th style="padding: 12px; text-align: center;">تاريخ التسليم</th>
                                    <th style="padding: 12px; text-align: center;">الملف</th>
                                    <th style="padding: 12px; text-align: center;">الملاحظة</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($submitted as $sub): ?>
                                <tr style="border-bottom: 1px solid #eee;">
                                    <td style="padding: 12px; border: 1px solid #ddd;"><?php echo htmlspecialchars($sub['homework_title']); ?></td>
                                    <td style="padding: 12px; border: 1px solid #ddd;"><?php echo htmlspecialchars($sub['class_name']); ?></td>
                                    <td style="padding: 12px; text-align: center; border: 1px solid #ddd;">
                                        <?php echo date('Y-m-d H:i', strtotime($sub['submitted_at'])); ?>
                                    </td>
                                    <td style="padding: 12px; text-align: center; border: 1px solid #ddd;">
                                        <a href="../assets/uploads/<?php echo htmlspecialchars($sub['file_path']); ?>" target="_blank" style="color: #007cba; text-decoration: underline;">عرض</a>
                                    </td>
                                    <td style="padding: 12px; text-align: center; border: 1px solid #ddd;">
                                        <?php echo htmlspecialchars($sub['note'] ?: '-'); ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p style="text-align: right; color: #777;">لا توجد واجبات مُسلَّمة.</p>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>

        <br>
        <a href="dashboard.php" class="btn" style="background: #6c757d; padding: 10px 20px; text-decoration: none; color: white; border-radius: 6px; display: inline-block;">
            <i class="fas fa-arrow-left"></i> العودة للرئيسية
        </a>
    </div>

    <script>
        // عرض اسم الملف عند اختياره
        document.querySelectorAll('.file-input-container input[type="file"]').forEach(input => {
            input.addEventListener('change', function(e) {
                const fileName = e.target.files[0]?.name || 'لم يتم اختيار ملف';
                const button = this.parentElement.querySelector('.file-input-button span');
                button.textContent = fileName;
            });
        });
    </script>
</body>
</html>