<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isStudent()) {
    header("Location: ../index.php");
    exit();
}

$student = $_SESSION['user'];

// جلب الصفوف المرتبطة بالطالب من جدول student_classes
$stmt = $pdo->prepare("
    SELECT c.id, c.name 
    FROM classes c 
    JOIN student_classes sc ON c.id = sc.class_id 
    WHERE sc.student_id = ?
");
$stmt->execute([$student['id']]);
$classes = $stmt->fetchAll();

if (empty($classes)) {
    die("<h3 style='text-align: center; color: red; margin-top: 50px;'>عذرًا، لم يتم تعيينك في أي صف بعد. تواصل مع المُدرِّس.</h3>");
}

if ($student['status'] == 'banned') {
    die("<h3 style='text-align: center; color: red; margin-top: 50px;'>عذرًا، حسابك محظور.</h3>");
}

// جلب أسماء الصفوف لعرضها
$class_names = array_column($classes, 'name');
$class_ids = array_column($classes, 'id');
$class_display = count($class_names) > 1 ? 'متعدد الصفوف' : htmlspecialchars($class_names[0]);

// جلب عدد الدروس للطالب من جميع صفوفه — باستخدام FIND_IN_SET
$total_lessons = 0;
foreach ($class_ids as $cid) {
    $stmt = $pdo->prepare("SELECT COUNT(*) as cnt FROM lessons WHERE FIND_IN_SET(?, class_ids) > 0");
    $stmt->execute([$cid]);
    $total_lessons += $stmt->fetch()['cnt'];
}

// جلب عدد الدروس القادمة
$now = date('Y-m-d H:i:s');
$upcoming_lessons = 0;
foreach ($class_ids as $cid) {
    $stmt = $pdo->prepare("SELECT COUNT(*) as cnt FROM lessons WHERE FIND_IN_SET(?, class_ids) > 0 AND start_time > ?");
    $stmt->execute([$cid, $now]);
    $upcoming_lessons += $stmt->fetch()['cnt'];
}

// جلب المحاضرة القادمة (من أي صف)
$next_lesson = null;
foreach ($class_ids as $cid) {
    $stmt = $pdo->prepare("SELECT l.*, c.name as class_name, t.name as teacher_name 
                           FROM lessons l 
                           JOIN classes c ON FIND_IN_SET(c.id, l.class_ids) > 0 
                           LEFT JOIN teachers t ON l.teacher_id = t.id
                           WHERE FIND_IN_SET(?, l.class_ids) > 0 AND l.start_time > ? 
                           ORDER BY l.start_time ASC LIMIT 1");
    $stmt->execute([$cid, $now]);
    $lesson = $stmt->fetch();
    if ($lesson && (!$next_lesson || strtotime($lesson['start_time']) < strtotime($next_lesson['start_time']))) {
        $next_lesson = $lesson;
    }
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>لوحة الطالب - اشراق</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        /* شريط التنقل العلوي */
        .top-nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding: 15px 25px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-weight: 700;
            font-size: 20px;
            color: var(--primary);
        }

        .nav-links {
            display: flex;
            gap: 20px;
            align-items: center;
        }

        .nav-link {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 10px 15px;
            border-radius: var(--border-radius);
            color: var(--dark);
            text-decoration: none;
            transition: var(--transition);
        }

        .nav-link:hover {
            background: var(--light);
            color: var(--primary);
        }

        .nav-link.active {
            background: var(--primary);
            color: white;
        }

        .profile-btn {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 10px 20px;
            background: var(--primary);
            color: white;
            border: none;
            border-radius: var(--border-radius);
            cursor: pointer;
            transition: var(--transition);
            font-weight: 600;
        }

        .profile-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        /* رأس الصفحة */
        .student-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            display: flex;
            align-items: center;
            gap: 20px;
            flex-wrap: wrap;
            position: relative;
            overflow: hidden;
        }

        .student-header::before {
            content: "";
            position: absolute;
            top: -50%;
            right: -50%;
            width: 100%;
            height: 200%;
            background: rgba(255, 255, 255, 0.1);
            transform: rotate(45deg);
        }

        .student-avatar {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            border: 4px solid white;
            object-fit: cover;
            box-shadow: 0 4px 10px rgba(0,0,0,0.2);
            position: relative;
            z-index: 1;
        }

        .student-info {
            flex: 1;
            position: relative;
            z-index: 1;
        }

        .student-info h2 {
            font-size: 28px;
            margin-bottom: 5px;
        }

        .student-info p {
            margin: 5px 0;
            opacity: 0.9;
        }

        .student-status {
            display: flex;
            gap: 15px;
            margin-top: 15px;
        }

        .status-badge {
            padding: 8px 15px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 20px;
            font-size: 14px;
            display: flex;
            align-items: center;
            gap: 5px;
        }

        /* لوحة الإحصائيات */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            text-align: center;
            transition: var(--transition);
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            width: 5px;
            height: 100%;
            background: var(--primary);
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(0, 0, 0, 0.15);
        }

        .stat-icon {
            width: 70px;
            height: 70px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
            font-size: 28px;
        }

        .stat-card:nth-child(1) .stat-icon {
            background: rgba(67, 97, 238, 0.1);
            color: var(--primary);
        }

        .stat-card:nth-child(2) .stat-icon {
            background: rgba(114, 9, 183, 0.1);
            color: var(--secondary);
        }

        .stat-card:nth-child(3) .stat-icon {
            background: rgba(247, 37, 133, 0.1);
            color: var(--accent);
        }

        .stat-number {
            font-size: 36px;
            font-weight: 700;
            margin: 10px 0;
            color: var(--dark);
        }

        .stat-title {
            color: var(--gray);
            font-size: 16px;
        }

        /* كارد المحاضرة القادمة */
        .next-lesson-card {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
            border-left: 5px solid var(--primary);
            position: relative;
            overflow: hidden;
        }

        .next-lesson-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .next-lesson-title {
            font-size: 22px;
            color: var(--primary);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .countdown {
            background: var(--primary);
            color: white;
            padding: 12px 18px;
            border-radius: var(--border-radius);
            font-weight: bold;
            font-size: 18px;
            display: flex;
            align-items: center;
            gap: 10px;
            box-shadow: 0 4px 10px rgba(67, 97, 238, 0.3);
        }

        .next-lesson-content {
            display: grid;
            grid-template-columns: 1fr auto;
            gap: 20px;
            align-items: center;
        }

        .next-lesson-info h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--dark);
        }

        .next-lesson-info p {
            color: var(--gray);
            margin-bottom: 15px;
            line-height: 1.6;
        }

        .next-lesson-meta {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 14px;
            color: var(--gray);
            background: var(--light);
            padding: 8px 12px;
            border-radius: var(--border-radius);
        }

        .join-btn {
            background: var(--primary);
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: var(--border-radius);
            font-weight: 600;
            font-size: 16px;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
            box-shadow: 0 4px 10px rgba(67, 97, 238, 0.3);
        }

        .join-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: 0 6px 15px rgba(67, 97, 238, 0.4);
        }

        .join-btn:disabled {
            background: #ccc;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }

        /* شريط البحث والتحكم */
        .control-bar {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            margin-bottom: 20px;
            align-items: center;
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .search-box {
            flex: 1;
            min-width: 250px;
            position: relative;
        }

        .search-box input {
            width: 100%;
            padding: 15px 50px 15px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .search-box input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray);
        }

        .filter-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .filter-btn {
            padding: 12px 18px;
            background: #f8f9fa;
            border: 1px solid #ddd;
            border-radius: var(--border-radius);
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .filter-btn.active {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }

        /* قائمة الدروس */
        .lessons-section {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
        }

        .section-title {
            font-size: 22px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .refresh-btn {
            background: #f8f9fa;
            border: 1px solid #ddd;
            padding: 10px 18px;
            border-radius: var(--border-radius);
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 5px;
            transition: var(--transition);
        }

        .refresh-btn:hover {
            background: #e9ecef;
        }

        #lessonsContainer {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
        }

        .lesson-card {
            background: white;
            border: 1px solid #eee;
            border-radius: var(--border-radius);
            padding: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
            transition: var(--transition);
            display: flex;
            flex-direction: column;
            position: relative;
            overflow: hidden;
        }

        .lesson-card::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary);
        }

        .lesson-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.1);
        }

        .lesson-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }

        .lesson-title {
            font-size: 18px;
            color: var(--primary);
            margin-bottom: 5px;
        }

        .lesson-type {
            background: var(--primary);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }

        .lesson-type.recorded {
            background: var(--secondary);
        }

        .lesson-description {
            color: var(--gray);
            margin-bottom: 15px;
            line-height: 1.5;
            flex-grow: 1;
        }

        .lesson-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 15px;
        }

        .lesson-meta div {
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .lesson-actions {
            margin-top: auto;
        }

        .lesson-btn {
            display: block;
            text-align: center;
            padding: 12px;
            border-radius: var(--border-radius);
            text-decoration: none;
            font-weight: 600;
            transition: var(--transition);
        }

        .lesson-btn.live {
            background: var(--primary);
            color: white;
        }

        .lesson-btn.live:hover {
            background: var(--primary-dark);
        }

        .lesson-btn.recorded {
            background: var(--secondary);
            color: white;
        }

        .lesson-btn.recorded:hover {
            background: #5e0d9b;
        }

        .lesson-btn.disabled {
            background: #ccc;
            color: #777;
            cursor: not-allowed;
        }

        /* التذييل */
        .last-update {
            text-align: center;
            color: var(--gray);
            font-size: 14px;
            margin-top: 30px;
            padding: 15px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .top-nav {
                flex-direction: column;
                gap: 15px;
            }
            
            .nav-links {
                width: 100%;
                justify-content: center;
                flex-wrap: wrap;
            }
            
            .student-header {
                flex-direction: column;
                text-align: center;
            }
            
            .student-status {
                justify-content: center;
            }
            
            .control-bar {
                flex-direction: column;
            }
            
            .search-box {
                width: 100%;
            }
            
            #lessonsContainer {
                grid-template-columns: 1fr;
            }
            
            .section-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .next-lesson-content {
                grid-template-columns: 1fr;
                text-align: center;
            }
            
            .countdown {
                font-size: 16px;
                padding: 10px 15px;
                justify-content: center;
            }
            
            .next-lesson-meta {
                justify-content: center;
            }
            
            .join-btn {
                width: 100%;
                justify-content: center;
            }
        }

        /* الرسوم المتحركة */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .lesson-card {
            animation: fadeIn 0.5s ease;
        }

        .pulse {
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }
        
        .blink {
            animation: blink 1s infinite;
        }
        
        @keyframes blink {
            0% { opacity: 1; }
            50% { opacity: 0.7; }
            100% { opacity: 1; }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- شريط التنقل العلوي -->
        <div class="top-nav">
            <div class="logo">
                <i class="fas fa-graduation-cap"></i>
                <span>اشراق</span>
            </div>
            
            <div class="nav-links">
                <a href="#" class="nav-link active">
                    <i class="fas fa-home"></i>
                    الرئيسية
                </a>
                <a href="lessons.php" class="nav-link">
                    <i class="fas fa-book-open"></i>
                    الدروس
                </a>
                <a href="assignments.php" class="nav-link">
                    <i class="fas fa-tasks"></i>
                    الواجبات
                </a>
                <a href="materials.php" class="nav-link ">
                    <i class="fas fa-file"></i>
                    مادة تعليمية
                </a>
                <a href="tasks.php" class="nav-link">
                    <i class="fas fa-tasks"></i>
                    مهامي
                </a>
                
            </div>
            
            <button class="profile-btn" onclick="window.location.href='profile.php'">
                <i class="fas fa-user-circle"></i>
                الملف الشخصي
            </button>
        </div>

        <!-- رأس الطالب -->
        <div class="student-header">
            <img src="../assets/<?php echo htmlspecialchars($student['image_url']); ?>" 
                 alt="صورة الطالب" 
                 class="student-avatar">
            
            <div class="student-info">
                <h2><?php echo htmlspecialchars($student['name']); ?></h2>
                <p>طالب في: <strong><?php echo $class_display; ?></strong></p>
                <p><?php echo htmlspecialchars($student['email']); ?></p>
                
                <div class="student-status">
                    <div class="status-badge">
                        <i class="fas fa-book-open"></i>
                        <span>الدروس: <?php echo $total_lessons; ?></span>
                    </div>
                    <div class="status-badge">
                        <i class="fas fa-clock"></i>
                        <span>القادمة: <?php echo $upcoming_lessons; ?></span>
                    </div>
                    <?php if (count($class_names) > 1): ?>
                    <div class="status-badge">
                        <i class="fas fa-layer-group"></i>
                        <span><?php echo count($class_names); ?> صفوف</span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- إحصائيات سريعة -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-video"></i>
                </div>
                <div class="stat-number" id="liveLessonsCount">0</div>
                <div class="stat-title">دروس مباشرة</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-play-circle"></i>
                </div>
                <div class="stat-number" id="recordedLessonsCount">0</div>
                <div class="stat-title">دروس مسجلة</div>
            </div>
            
            <div class="stat-card" onclick="location.href='tasks.php'" style="cursor: pointer;">
                <div class="stat-icon">
                    <i class="fas fa-tasks"></i>
                </div>
                <div class="stat-number"></div>
                <div class="stat-title"> مهامي</div>
            </div>
        </div>

        <!-- المحاضرة القادمة -->
        <?php if ($next_lesson): ?>
        <div class="next-lesson-card">
            <div class="next-lesson-header">
                <h2 class="next-lesson-title">
                    <i class="fas fa-calendar-alt"></i>
                    المحاضرة القادمة
                </h2>
                <div class="countdown" id="nextLessonCountdown">
                    <i class="fas fa-clock"></i>
                    <span>--:--:--</span>
                </div>
            </div>
            
            <div class="next-lesson-content">
                <div class="next-lesson-info">
                    <h3><?php echo htmlspecialchars($next_lesson['title']); ?></h3>
                    <p><?php echo htmlspecialchars($next_lesson['description']); ?></p>
                    
                    <div class="next-lesson-meta">
                        <div class="meta-item">
                            <i class="fas fa-clock"></i>
                            <span><?php echo date('Y-m-d H:i', strtotime($next_lesson['start_time'])); ?></span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-chalkboard-teacher"></i>
                            <span><?php echo htmlspecialchars($next_lesson['teacher_name'] ?? 'غير معروف'); ?></span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-layer-group"></i>
                            <span><?php echo htmlspecialchars($next_lesson['class_name']); ?></span>
                        </div>
                    </div>
                </div>
                
                <a href="<?php echo htmlspecialchars($next_lesson['zoom_link']); ?>" target="_blank" class="join-btn" id="joinNextLessonBtn">
                    <i class="fas fa-video"></i>
                    انضم إلى المحاضرة
                </a>
            </div>
        </div>
        <?php endif; ?>

        <!-- شريط البحث والتحكم -->
        <div class="control-bar">
            <div class="search-box">
                <input type="text" id="searchLessons" placeholder="ابحث عن درس أو محتوى...">
                <i class="fas fa-search"></i>
            </div>
            
            <div class="filter-buttons">
                <button class="filter-btn active" data-filter="all">
                    <i class="fas fa-layer-group"></i>
                    الكل
                </button>
                <button class="filter-btn" data-filter="live">
                    <i class="fas fa-video"></i>
                    مباشر
                </button>
                <button class="filter-btn" data-filter="recorded">
                    <i class="fas fa-play-circle"></i>
                    مسجل
                </button>
                <button class="filter-btn" data-filter="upcoming">
                    <i class="fas fa-clock"></i>
                    قادم
                </button>
            </div>
        </div>

        <!-- قائمة الدروس -->
        <div class="lessons-section">
            <div class="section-header">
                <h3 class="section-title">
                    <i class="fas fa-book-open"></i>
                    الدروس المتاحة
                </h3>
                
                <button class="refresh-btn" id="refreshLessons">
                    <i class="fas fa-sync-alt"></i>
                    تحديث
                </button>
            </div>

            <div id="lessonsContainer">
                <p style="text-align: center; color: #777; grid-column: 1 / -1;">
                    جاري تحميل الدروس...
                </p>
            </div>
        </div>

        <!-- التذييل -->
        <div class="last-update" id="lastUpdate">
            <i class="fas fa-clock"></i>
            آخر تحديث: الآن
        </div>
    </div>

    <script>
    const lessonsContainer = document.getElementById('lessonsContainer');
    const liveLessonsCount = document.getElementById('liveLessonsCount');
    const recordedLessonsCount = document.getElementById('recordedLessonsCount');
    const lastUpdate = document.getElementById('lastUpdate');
    const searchInput = document.getElementById('searchLessons');
    const refreshBtn = document.getElementById('refreshLessons');
    const filterButtons = document.querySelectorAll('.filter-btn');
    const nextLessonCountdown = document.getElementById('nextLessonCountdown');
    const joinNextLessonBtn = document.getElementById('joinNextLessonBtn');
    
    let allLessons = [];
    let currentFilter = 'all';
    let nextLessonTime = <?php echo $next_lesson ? strtotime($next_lesson['start_time']) * 1000 : 'null'; ?>;

    // عدّاد تنازلي للمحاضرة القادمة
    function updateCountdown() {
        if (!nextLessonTime) return;
        
        const now = new Date().getTime();
        const distance = nextLessonTime - now;
        
        if (distance < 0) {
            nextLessonCountdown.innerHTML = '<i class="fas fa-play-circle"></i> <span>بدأت المحاضرة</span>';
            if (joinNextLessonBtn) {
                joinNextLessonBtn.classList.remove('blink');
            }
            return;
        }
        
        // حساب الأيام، الساعات، الدقائق والثواني
        const days = Math.floor(distance / (1000 * 60 * 60 * 24));
        const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((distance % (1000 * 60)) / 1000);
        
        // عرض النتيجة
        let countdownText = '';
        if (days > 0) {
            countdownText = `${days} يوم ${hours} ساعة ${minutes} دقيقة`;
        } else if (hours > 0) {
            countdownText = `${hours} ساعة ${minutes} دقيقة ${seconds} ثانية`;
        } else {
            countdownText = `${minutes} دقيقة ${seconds} ثانية`;
            
            // إذا بقي أقل من 5 دقائق، أضف تأثيرًا مرئيًا
            if (minutes < 5 && joinNextLessonBtn) {
                joinNextLessonBtn.classList.add('blink');
            }
        }
        
        nextLessonCountdown.innerHTML = `<i class="fas fa-clock"></i> <span>${countdownText}</span>`;
    }

    // جلب الدروس — الآن من API يدعم عدة صفوف
    function fetchLessons() {
        refreshBtn.classList.add('pulse');
        
        // يمكنك تعديل API ليقبل class_ids أو يجلب حسب الطالب
        fetch('../api/get_lessons.php?student_id=<?php echo $student['id']; ?>')
            .then(res => res.json())
            .then(data => {
                if (data.error) {
                    lessonsContainer.innerHTML = `<p style="color: red; text-align: center; grid-column: 1 / -1;">${data.error}</p>`;
                    return;
                }

                allLessons = data.lessons;
                renderLessons(allLessons);
                updateStats(allLessons);
                lastUpdate.innerHTML = `<i class="fas fa-clock"></i> آخر تحديث: ${new Date().toLocaleTimeString()}`;
                
                refreshBtn.classList.remove('pulse');
            })
            .catch(err => {
                lessonsContainer.innerHTML = `<p style="color: red; text-align: center; grid-column: 1 / -1;">فشل الاتصال بالخادم.</p>`;
                refreshBtn.classList.remove('pulse');
            });
    }

    // عرض الدروس
    function renderLessons(lessons) {
        if (lessons.length === 0) {
            lessonsContainer.innerHTML = `
                <p style="text-align: center; color: #777; grid-column: 1 / -1; padding: 40px;">
                    لا توجد دروس متاحة حاليًا.
                </p>`;
            return;
        }

        let html = '';
        lessons.forEach(lesson => {
            if (shouldFilterLesson(lesson)) return;
            
            const now = new Date();
            const startTime = lesson.start_time ? new Date(lesson.start_time) : null;
            const isFuture = startTime && now < startTime;
            const isLive = lesson.lesson_type === 'live';

            let buttonHtml = '';
            if (isLive) {
                if (isFuture) {
                    buttonHtml = `
                        <button class="lesson-btn disabled" disabled>
                            <i class="fas fa-clock"></i> يبدأ بعد: ${formatTimeLeft(lesson.start_time)}
                        </button>`;
                } else {
                    buttonHtml = `
                        <a href="${lesson.zoom_link}" target="_blank" class="lesson-btn live">
                            <i class="fas fa-play-circle"></i> انضم إلى المحاضرة
                        </a>`;
                }
            } else {
                buttonHtml = `
                    <a href="${lesson.recorded_link}" target="_blank" class="lesson-btn recorded">
                        <i class="fas fa-video"></i> شاهد التسجيل
                    </a>`;
            }

            html += `
            <div class="lesson-card" 
                 data-title="${lesson.title.toLowerCase()}" 
                 data-desc="${lesson.description.toLowerCase()}"
                 data-type="${lesson.lesson_type}"
                 data-status="${isFuture ? 'upcoming' : 'available'}">
                
                <div class="lesson-header">
                    <div>
                        <h4 class="lesson-title">${lesson.title}</h4>
                    </div>
                    <div class="lesson-type ${isLive ? '' : 'recorded'}">
                        ${isLive ? 'مباشر' : 'مسجل'}
                    </div>
                </div>
                
                <p class="lesson-description">${lesson.description}</p>
                
                <div class="lesson-meta">
                    ${startTime ? `
                    <div>
                        <i class="fas fa-clock"></i>
                        <span>${formatDateTime(startTime)}</span>
                    </div>
                    ` : ''}
                    
                    <div>
                        <i class="fas fa-chalkboard-teacher"></i>
                        <span>${lesson.teacher_name || 'غير معروف'}</span>
                    </div>
                    
                    <div>
                        <i class="fas fa-layer-group"></i>
                        <span>${lesson.class_name || 'غير معروف'}</span>
                    </div>
                </div>
                
                <div class="lesson-actions">
                    ${buttonHtml}
                </div>
            </div>`;
        });

        lessonsContainer.innerHTML = html;
    }

    // تحديث الإحصائيات
    function updateStats(lessons) {
        const liveLessons = lessons.filter(l => l.lesson_type === 'live').length;
        const recordedLessons = lessons.filter(l => l.lesson_type === 'recorded').length;
        
        liveLessonsCount.textContent = liveLessons;
        recordedLessonsCount.textContent = recordedLessons;
    }

    // التحقق من تطبيق الفلتر
    function shouldFilterLesson(lesson) {
        if (currentFilter === 'all') return false;
        if (currentFilter === 'live' && lesson.lesson_type !== 'live') return true;
        if (currentFilter === 'recorded' && lesson.lesson_type !== 'recorded') return true;
        if (currentFilter === 'upcoming') {
            const startTime = lesson.start_time ? new Date(lesson.start_time) : null;
            return !startTime || new Date() >= startTime;
        }
        return false;
    }

    // تنسيق التاريخ والوقت
    function formatDateTime(dateTime) {
        return new Date(dateTime).toLocaleString('ar', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    // المؤقت التنازلي
    function formatTimeLeft(dateTimeStr) {
        const end = new Date(dateTimeStr).getTime();
        const now = new Date().getTime();
        const gap = end - now;

        if (gap <= 0) return "الآن";

        const days = Math.floor(gap / (1000 * 60 * 60 * 24));
        const hours = Math.floor((gap % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((gap % (1000 * 60 * 60)) / (1000 * 60));
        
        if (days > 0) return `${days} يوم ${hours} ساعة`;
        if (hours > 0) return `${hours} ساعة ${minutes} دقيقة`;
        return `${minutes} دقيقة`;
    }

    // البحث الفوري
    searchInput.addEventListener('input', function() {
        const filter = this.value.toLowerCase();
        const cards = document.querySelectorAll('.lesson-card');
        let visibleCount = 0;

        cards.forEach(card => {
            const title = card.dataset.title || '';
            const desc = card.dataset.desc || '';
            
            if (title.includes(filter) || desc.includes(filter)) {
                card.style.display = 'flex';
                visibleCount++;
            } else {
                card.style.display = 'none';
            }
        });

        if (visibleCount === 0) {
            lessonsContainer.innerHTML = `
                <p style="text-align: center; color: #777; grid-column: 1 / -1; padding: 40px;">
                    لا توجد نتائج بحث.
                </p>`;
        }
    });

    // أزرار التصفية
    filterButtons.forEach(btn => {
        btn.addEventListener('click', function() {
            filterButtons.forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            currentFilter = this.dataset.filter;
            renderLessons(allLessons);
        });
    });

    // زر التحديث
    refreshBtn.addEventListener('click', fetchLessons);

    // أول جلب + تحديث كل 30 ثانية
    fetchLessons();
    setInterval(fetchLessons, 30000);
    
    // تحديث العدّاد التنازلي كل ثانية
    if (nextLessonTime) {
        updateCountdown();
        setInterval(updateCountdown, 1000);
    }
    </script>
</body>
</html>