<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isStudent()) {
    header("Location: ../index.php");
    exit();
}

$student = $_SESSION['user'];

// جلب الصفوف المرتبطة بالطالب من جدول student_classes
$stmt = $pdo->prepare("
    SELECT c.id, c.name 
    FROM classes c 
    JOIN student_classes sc ON c.id = sc.class_id 
    WHERE sc.student_id = ?
");
$stmt->execute([$student['id']]);
$classes = $stmt->fetchAll();

if (empty($classes)) {
    die("<h3 style='text-align: center; color: red; margin-top: 50px;'>عذرًا، لم يتم تعيينك في أي صف بعد.</h3>");
}

if ($student['status'] == 'banned') {
    die("<h3 style='text-align: center; color: red; margin-top: 50px;'>عذرًا، حسابك محظور.</h3>");
}

// جلب أسماء الصفوف و IDs لاستخدامها في استعلامات الدروس
$class_ids = array_column($classes, 'id');
$placeholders = str_repeat('?,', count($class_ids) - 1) . '?';
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>دروسي - اشراق</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        /* شريط التنقل العلوي */
        .top-nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding: 15px 25px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-weight: 700;
            font-size: 20px;
            color: var(--primary);
        }

        .nav-links {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        .nav-link {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 15px;
            border-radius: var(--border-radius);
            color: var(--dark);
            text-decoration: none;
            transition: var(--transition);
        }

        .nav-link:hover {
            background: var(--light);
            color: var(--primary);
        }

        .nav-link.active {
            background: var(--primary);
            color: white;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        .page-header p {
            opacity: 0.9;
            font-size: 18px;
        }

        /* عدادات */
        .counters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .counter-card {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            text-align: center;
            transition: var(--transition);
        }

        .counter-card:hover {
            transform: translateY(-5px);
        }

        .counter-icon {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
            font-size: 24px;
        }

        .counter-card:nth-child(1) .counter-icon {
            background: rgba(67, 97, 238, 0.1);
            color: var(--primary);
        }

        .counter-card:nth-child(2) .counter-icon {
            background: rgba(114, 9, 183, 0.1);
            color: var(--secondary);
        }

        .counter-card:nth-child(3) .counter-icon {
            background: rgba(247, 37, 133, 0.1);
            color: var(--accent);
        }

        .counter-number {
            font-size: 32px;
            font-weight: 700;
            margin: 10px 0;
            color: var(--dark);
        }

        .counter-title {
            color: var(--gray);
            font-size: 16px;
        }

        /* شريط البحث */
        .search-section {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .search-box {
            position: relative;
        }

        .search-box input {
            width: 100%;
            padding: 15px 50px 15px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .search-box input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray);
        }

        /* قائمة الدروس */
        .lessons-section {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
        }

        .section-title {
            font-size: 22px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .refresh-btn {
            background: #f8f9fa;
            border: 1px solid #ddd;
            padding: 10px 18px;
            border-radius: var(--border-radius);
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 5px;
            transition: var(--transition);
        }

        .refresh-btn:hover {
            background: #e9ecef;
        }

        #lessonsContainer {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
        }

        .lesson-card {
            background: white;
            border: 1px solid #eee;
            border-radius: var(--border-radius);
            padding: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
            transition: var(--transition);
            display: flex;
            flex-direction: column;
            position: relative;
            overflow: hidden;
        }

        .lesson-card::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary);
        }

        .lesson-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.1);
        }

        .lesson-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }

        .lesson-title {
            font-size: 18px;
            color: var(--primary);
            margin-bottom: 5px;
        }

        .lesson-type {
            background: var(--primary);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }

        .lesson-type.recorded {
            background: var(--secondary);
        }

        .lesson-description {
            color: var(--gray);
            margin-bottom: 15px;
            line-height: 1.5;
            flex-grow: 1;
        }

        .lesson-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 15px;
        }

        .lesson-meta div {
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .lesson-actions {
            margin-top: auto;
        }

        .lesson-btn {
            display: block;
            text-align: center;
            padding: 12px;
            border-radius: var(--border-radius);
            text-decoration: none;
            font-weight: 600;
            transition: var(--transition);
        }

        .lesson-btn.live {
            background: var(--primary);
            color: white;
        }

        .lesson-btn.live:hover {
            background: var(--primary-dark);
        }

        .lesson-btn.recorded {
            background: var(--secondary);
            color: white;
        }

        .lesson-btn.recorded:hover {
            background: #5e0d9b;
        }

        .lesson-btn.disabled {
            background: #ccc;
            color: #777;
            cursor: not-allowed;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: var(--gray);
            grid-column: 1 / -1;
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        /* التذييل */
        .last-update {
            text-align: center;
            color: var(--gray);
            font-size: 14px;
            margin-top: 30px;
            padding: 15px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .top-nav {
                flex-direction: column;
                gap: 15px;
            }
            
            .nav-links {
                width: 100%;
                justify-content: center;
                flex-wrap: wrap;
            }
            
            .counters-grid {
                grid-template-columns: 1fr;
            }
            
            #lessonsContainer {
                grid-template-columns: 1fr;
            }
            
            .section-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }

        /* الرسوم المتحركة */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .lesson-card {
            animation: fadeIn 0.5s ease;
        }

        .pulse {
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- شريط التنقل العلوي -->
        <div class="top-nav">
            <div class="logo">
                <i class="fas fa-graduation-cap"></i>
                <span>اشراق</span>
            </div>
            
            <div class="nav-links">
                <a href="dashboard.php" class="nav-link">
                    <i class="fas fa-home"></i>
                    الرئيسية
                </a>
                <a href="lessons.php" class="nav-link active">
                    <i class="fas fa-book-open"></i>
                    الدروس
                </a>
                <a href="tasks.php" class="nav-link">
                    <i class="fas fa-tasks"></i>
                    المهام
                </a>
                <a href="profile.php" class="nav-link">
                    <i class="fas fa-user"></i>
                    الملف الشخصي
                </a>
            </div>
        </div>

        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1><i class="fas fa-book-open"></i> دروسي</h1>
            <p>الدروس من جميع صفوفك</p>
        </div>

        <!-- عدادات -->
        <div class="counters-grid">
            <div class="counter-card">
                <div class="counter-icon">
                    <i class="fas fa-book"></i>
                </div>
                <div class="counter-number" id="totalCounter">0</div>
                <div class="counter-title">إجمالي الدروس</div>
            </div>
            
            <div class="counter-card">
                <div class="counter-icon">
                    <i class="fas fa-video"></i>
                </div>
                <div class="counter-number" id="liveCounter">0</div>
                <div class="counter-title">دروس مباشرة</div>
            </div>
            
            <div class="counter-card">
                <div class="counter-icon">
                    <i class="fas fa-play-circle"></i>
                </div>
                <div class="counter-number" id="recordedCounter">0</div>
                <div class="counter-title">دروس مسجلة</div>
            </div>
        </div>

        <!-- شريط البحث -->
        <div class="search-section">
            <div class="search-box">
                <input type="text" id="searchLessons" placeholder="ابحث عن درس أو محتوى...">
                <i class="fas fa-search"></i>
            </div>
        </div>

        <!-- قائمة الدروس -->
        <div class="lessons-section">
            <div class="section-header">
                <h3 class="section-title">
                    <i class="fas fa-list"></i>
                    الدروس المتاحة
                </h3>
                
                <button class="refresh-btn" id="refreshLessons">
                    <i class="fas fa-sync-alt"></i>
                    تحديث
                </button>
            </div>

            <div id="lessonsContainer">
                <div class="empty-state">
                    <i class="fas fa-book-open"></i>
                    <p>جاري تحميل الدروس...</p>
                </div>
            </div>
        </div>

        <!-- التذييل -->
        <div class="last-update" id="lastUpdate">
            <i class="fas fa-clock"></i>
            آخر تحديث: الآن
        </div>
    </div>

    <script>
    const lessonsContainer = document.getElementById('lessonsContainer');
    const totalCounter = document.getElementById('totalCounter');
    const liveCounter = document.getElementById('liveCounter');
    const recordedCounter = document.getElementById('recordedCounter');
    const lastUpdate = document.getElementById('lastUpdate');
    const searchInput = document.getElementById('searchLessons');
    const refreshBtn = document.getElementById('refreshLessons');
    
    let allLessons = [];

    // جلب الدروس — الآن من API يدعم عدة صفوف
    function fetchLessons() {
        refreshBtn.classList.add('pulse');
        
        // يمكنك تعديل API ليقبل student_id أو يجلب حسب الطالب
        fetch('../api/get_lessons.php?student_id=<?php echo $student['id']; ?>')
            .then(res => res.json())
            .then(data => {
                if (data.error) {
                    lessonsContainer.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-exclamation-circle"></i>
                            <p>${data.error}</p>
                        </div>`;
                    return;
                }

                allLessons = data.lessons;
                renderLessons(allLessons);
                updateCounters(allLessons);
                lastUpdate.innerHTML = `<i class="fas fa-clock"></i> آخر تحديث: ${new Date().toLocaleTimeString()}`;
                
                refreshBtn.classList.remove('pulse');
            })
            .catch(err => {
                lessonsContainer.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-exclamation-triangle"></i>
                        <p>فشل الاتصال بالخادم</p>
                    </div>`;
                refreshBtn.classList.remove('pulse');
            });
    }

    // عرض الدروس
    function renderLessons(lessons) {
        if (lessons.length === 0) {
            lessonsContainer.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-book-open"></i>
                    <p>لا توجد دروس متاحة حاليًا</p>
                </div>`;
            return;
        }

        let html = '';
        lessons.forEach(lesson => {
            const now = new Date();
            const startTime = lesson.start_time ? new Date(lesson.start_time) : null;
            const isFuture = startTime && now < startTime;
            const isLive = lesson.lesson_type === 'live';

            let buttonHtml = '';
            if (isLive) {
                if (isFuture) {
                    buttonHtml = `
                        <button class="lesson-btn disabled" disabled>
                            <i class="fas fa-clock"></i> يبدأ بعد: ${formatTimeLeft(lesson.start_time)}
                        </button>`;
                } else {
                    buttonHtml = `
                        <a href="${lesson.zoom_link}" target="_blank" class="lesson-btn live">
                            <i class="fas fa-play-circle"></i> انضم إلى المحاضرة
                        </a>`;
                }
            } else {
                buttonHtml = `
                    <a href="${lesson.recorded_link}" target="_blank" class="lesson-btn recorded">
                        <i class="fas fa-video"></i> شاهد التسجيل
                    </a>`;
            }

            html += `
            <div class="lesson-card" 
                 data-title="${lesson.title.toLowerCase()}" 
                 data-desc="${lesson.description.toLowerCase()}"
                 data-type="${lesson.lesson_type}">
                
                <div class="lesson-header">
                    <div>
                        <h4 class="lesson-title">${lesson.title}</h4>
                    </div>
                    <div class="lesson-type ${isLive ? '' : 'recorded'}">
                        ${isLive ? 'مباشر' : 'مسجل'}
                    </div>
                </div>
                
                <p class="lesson-description">${lesson.description}</p>
                
                <div class="lesson-meta">
                    ${startTime ? `
                    <div>
                        <i class="fas fa-clock"></i>
                        <span>${formatDateTime(startTime)}</span>
                    </div>
                    ` : ''}
                    
                    <div>
                        <i class="fas fa-chalkboard-teacher"></i>
                        <span>${lesson.teacher_name || 'غير معروف'}</span>
                    </div>
                    
                    <div>
                        <i class="fas fa-layer-group"></i>
                        <span>${lesson.class_name || 'غير معروف'}</span>
                    </div>
                </div>
                
                <div class="lesson-actions">
                    ${buttonHtml}
                </div>
            </div>`;
        });

        lessonsContainer.innerHTML = html;
    }

    // تحديث العدادات
    function updateCounters(lessons) {
        const liveLessons = lessons.filter(l => l.lesson_type === 'live').length;
        const recordedLessons = lessons.filter(l => l.lesson_type === 'recorded').length;
        
        totalCounter.textContent = lessons.length;
        liveCounter.textContent = liveLessons;
        recordedCounter.textContent = recordedLessons;
    }

    // تنسيق التاريخ والوقت
    function formatDateTime(dateTime) {
        return new Date(dateTime).toLocaleString('ar', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    // المؤقت التنازلي
    function formatTimeLeft(dateTimeStr) {
        const end = new Date(dateTimeStr).getTime();
        const now = new Date().getTime();
        const gap = end - now;

        if (gap <= 0) return "الآن";

        const days = Math.floor(gap / (1000 * 60 * 60 * 24));
        const hours = Math.floor((gap % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((gap % (1000 * 60 * 60)) / (1000 * 60));
        
        if (days > 0) return `${days} يوم ${hours} ساعة`;
        if (hours > 0) return `${hours} ساعة ${minutes} دقيقة`;
        return `${minutes} دقيقة`;
    }

    // البحث الفوري
    searchInput.addEventListener('input', function() {
        const filter = this.value.toLowerCase();
        const cards = document.querySelectorAll('.lesson-card');
        let visibleCount = 0;

        cards.forEach(card => {
            const title = card.dataset.title || '';
            const desc = card.dataset.desc || '';
            
            if (title.includes(filter) || desc.includes(filter)) {
                card.style.display = 'flex';
                visibleCount++;
            } else {
                card.style.display = 'none';
            }
        });

        if (visibleCount === 0) {
            lessonsContainer.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-search"></i>
                    <p>لا توجد نتائج بحث</p>
                </div>`;
        }
    });

    // زر التحديث
    refreshBtn.addEventListener('click', fetchLessons);

    // أول جلب + تحديث كل 30 ثانية
    fetchLessons();
    setInterval(fetchLessons, 30000);
    </script>
</body>
</html>