<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isStudent()) {
    header("Location: ../index.php");
    exit();
}

$student = $_SESSION['user'];
$student_id = $student['id'];

// جلب المواد المرتبطة بهذا الطالب فقط
$stmt = $pdo->prepare("
    SELECT m.*, t.name as teacher_name 
    FROM materials m
    JOIN material_students ms ON m.id = ms.material_id
    LEFT JOIN teachers t ON m.uploaded_by = t.id
    WHERE ms.student_id = ?
    ORDER BY m.uploaded_at DESC
");
$stmt->execute([$student_id]);
$materials = $stmt->fetchAll();

// حساب الإحصائيات
$total = count($materials);
$pdf_count = 0;
$image_count = 0;

foreach ($materials as $m) {
    if ($m['file_type'] == 'pdf') $pdf_count++;
    elseif ($m['file_type'] == 'image') $image_count++;
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مواقعي التعليمية - منصة التعليم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        /* شريط التنقل العلوي */
        .top-nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding: 15px 25px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-weight: 700;
            font-size: 20px;
            color: var(--primary);
        }

        .nav-links {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        .nav-link {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 15px;
            border-radius: var(--border-radius);
            color: var(--dark);
            text-decoration: none;
            transition: var(--transition);
        }

        .nav-link:hover {
            background: var(--light);
            color: var(--primary);
        }

        .nav-link.active {
            background: var(--primary);
            color: white;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        .page-header p {
            opacity: 0.9;
            font-size: 18px;
        }

        /* عدادات */
        .counters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .counter-card {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            text-align: center;
            transition: var(--transition);
        }

        .counter-card:hover {
            transform: translateY(-5px);
        }

        .counter-icon {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
            font-size: 24px;
        }

        .counter-card:nth-child(1) .counter-icon {
            background: rgba(67, 97, 238, 0.1);
            color: var(--primary);
        }

        .counter-card:nth-child(2) .counter-icon {
            background: rgba(211, 47, 47, 0.1);
            color: #d32f2f;
        }

        .counter-card:nth-child(3) .counter-icon {
            background: rgba(76, 201, 240, 0.1);
            color: #4cc9f0;
        }

        .counter-number {
            font-size: 32px;
            font-weight: 700;
            margin: 10px 0;
            color: var(--dark);
        }

        .counter-title {
            color: var(--gray);
            font-size: 16px;
        }

        /* شريط البحث */
        .search-section {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .search-box {
            position: relative;
        }

        .search-box input {
            width: 100%;
            padding: 15px 50px 15px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .search-box input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray);
        }

        /* قائمة المواد */
        .materials-section {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
        }

        .section-title {
            font-size: 22px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        #materialsContainer {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }

        .material-card {
            background: white;
            border: 1px solid #eee;
            border-radius: var(--border-radius);
            padding: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
            transition: var(--transition);
            display: flex;
            flex-direction: column;
            position: relative;
            overflow: hidden;
        }

        .material-card::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary);
        }

        .material-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.1);
        }

        .material-icon {
            width: 80px;
            height: 80px;
            margin: 0 auto 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            font-size: 30px;
        }

        .material-icon.pdf {
            background: rgba(211, 47, 47, 0.1);
            color: #d32f2f;
        }

        .material-icon.image {
            background: rgba(76, 201, 240, 0.1);
            color: #4cc9f0;
        }

        .material-title {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 10px;
            color: var(--dark);
            text-align: center;
        }

        .material-meta {
            color: var(--gray);
            font-size: 14px;
            margin: 10px 0;
            text-align: center;
        }

        .material-actions {
            margin-top: auto;
            display: flex;
            flex-direction: column;
            gap: 10px;
            margin-top: 15px;
        }

        .btn {
            display: block;
            text-align: center;
            padding: 12px;
            border-radius: var(--border-radius);
            text-decoration: none;
            font-weight: 600;
            transition: var(--transition);
            border: none;
            cursor: pointer;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: var(--gray);
            grid-column: 1 / -1;
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        /* التذييل */
        .last-update {
            text-align: center;
            color: var(--gray);
            font-size: 14px;
            margin-top: 30px;
            padding: 15px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .top-nav {
                flex-direction: column;
                gap: 15px;
            }
            
            .nav-links {
                width: 100%;
                justify-content: center;
                flex-wrap: wrap;
            }
            
            .counters-grid {
                grid-template-columns: 1fr;
            }
            
            #materialsContainer {
                grid-template-columns: 1fr;
            }
            
            .section-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }

        /* الرسوم المتحركة */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .material-card {
            animation: fadeIn 0.5s ease;
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- شريط التنقل العلوي -->
        <div class="top-nav">
            <div class="logo">
                <i class="fas fa-graduation-cap"></i>
                <span>اشراق</span>
            </div>
            
            <div class="nav-links">
                <a href="dashboard.php" class="nav-link">
                    <i class="fas fa-home"></i>
                    الرئيسية
                </a>
                <a href="lessons.php" class="nav-link">
                    <i class="fas fa-book-open"></i>
                    الدروس
                </a>
                <a href="materials.php" class="nav-link active">
                    <i class="fas fa-file"></i>
                    مادة تعليمية
                </a>
                <a href="profile.php" class="nav-link">
                    <i class="fas fa-user"></i>
                    الملف الشخصي
                </a>
            </div>
        </div>

        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1><i class="fas fa-file"></i> مواضعي التعليمية</h1>
            <p>المواد التي خصصها لك المُدرِّس</p>
        </div>

        <!-- عدادات -->
        <div class="counters-grid">
            <div class="counter-card">
                <div class="counter-icon">
                    <i class="fas fa-file"></i>
                </div>
                <div class="counter-number"><?php echo $total; ?></div>
                <div class="counter-title">إجمالي المواد</div>
            </div>
            
            <div class="counter-card">
                <div class="counter-icon pdf">
                    <i class="fas fa-file-pdf"></i>
                </div>
                <div class="counter-number"><?php echo $pdf_count; ?></div>
                <div class="counter-title">ملفات PDF</div>
            </div>
            
            <div class="counter-card">
                <div class="counter-icon image">
                    <i class="fas fa-file-image"></i>
                </div>
                <div class="counter-number"><?php echo $image_count; ?></div>
                <div class="counter-title">صور</div>
            </div>
        </div>

        <!-- شريط البحث -->
        <div class="search-section">
            <div class="search-box">
                <input type="text" id="searchMaterials" placeholder="ابحث عن مادة أو عنوان...">
                <i class="fas fa-search"></i>
            </div>
        </div>

        <!-- قائمة المواد -->
        <div class="materials-section">
            <div class="section-header">
                <h3 class="section-title">
                    <i class="fas fa-list"></i>
                    المواد المتاحة
                </h3>
            </div>

            <div id="materialsContainer">
                <?php if (empty($materials)): ?>
                    <div class="empty-state">
                        <i class="fas fa-file"></i>
                        <p>لا توجد مواد مخصصة لك حاليًا.</p>
                        <p>سيقوم المُدرِّس برفع المواد لك قريبًا.</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($materials as $material): ?>
                        <div class="material-card" data-title="<?php echo htmlspecialchars(strtolower($material['title'])); ?>">
                            <div class="material-icon <?php echo $material['file_type']; ?>">
                                <?php if ($material['file_type'] == 'pdf'): ?>
                                    <i class="fas fa-file-pdf"></i>
                                <?php else: ?>
                                    <i class="fas fa-file-image"></i>
                                <?php endif; ?>
                            </div>
                            
                            <h3 class="material-title"><?php echo htmlspecialchars($material['title']); ?></h3>
                            
                            <div class="material-meta">
                                <p>رفع بواسطة: <?php echo htmlspecialchars($material['teacher_name'] ?? 'غير معروف'); ?></p>
                                <p>في: <?php echo date('Y-m-d', strtotime($material['uploaded_at'])); ?></p>
                            </div>
                            
                            <?php if (!empty($material['description'])): ?>
                                <p style="color: #555; font-size: 14px; text-align: center; margin: 10px 0;">
                                    <?php echo htmlspecialchars($material['description']); ?>
                                </p>
                            <?php endif; ?>
                            
                            <div class="material-actions">
                                <?php if ($material['file_type'] == 'image'): ?>
                                    <a href="../<?php echo htmlspecialchars($material['file_path']); ?>" 
                                       target="_blank" 
                                       class="btn btn-secondary">
                                        <i class="fas fa-eye"></i> عرض الصورة
                                    </a>
                                <?php endif; ?>
                                
                                <a href="../<?php echo htmlspecialchars($material['file_path']); ?>" 
                                   download 
                                   class="btn btn-primary">
                                    <i class="fas fa-download"></i> تنزيل
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- التذييل -->
        <div class="last-update">
            <i class="fas fa-clock"></i>
            آخر تحديث: <?php echo date('Y-m-d H:i'); ?>
        </div>
    </div>

    <script>
    // البحث التلقائي
    document.getElementById('searchMaterials').addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase().trim();
        const cards = document.querySelectorAll('.material-card');
        let visibleCount = 0;

        cards.forEach(card => {
            const title = card.dataset.title || '';
            
            if (title.includes(searchTerm)) {
                card.style.display = 'flex';
                visibleCount++;
            } else {
                card.style.display = 'none';
            }
        });
    });
    </script>
</body>
</html>