<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isStudent()) {
    header("Location: ../index.php");
    exit();
}

$student_id = $_SESSION['user']['id'];
$error = '';
$success = '';

// --- إضافة مهمة جديدة ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['add_task'])) {
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $due_date = !empty($_POST['due_date']) ? $_POST['due_date'] : null;

    if (empty($title)) {
        $error = "العنوان مطلوب.";
    } else {
        try {
            $stmt = $pdo->prepare("INSERT INTO student_tasks (student_id, title, description, due_date) VALUES (?, ?, ?, ?)");
            $stmt->execute([$student_id, $title, $description, $due_date]);
            $success = "تم إضافة المهمة بنجاح!";
        } catch (Exception $e) {
            $error = "حدث خطأ أثناء الحفظ.";
        }
    }
}

// --- تعديل حالة المهمة (مكتمل / غير مكتمل) ---
if (isset($_POST['toggle_task'])) {
    $task_id = $_POST['task_id'];
    $stmt = $pdo->prepare("SELECT id, is_completed FROM student_tasks WHERE id = ? AND student_id = ?");
    $stmt->execute([$task_id, $student_id]);
    $task = $stmt->fetch();

    if ($task) {
        $new_status = $task['is_completed'] ? 0 : 1;
        $stmt = $pdo->prepare("UPDATE student_tasks SET is_completed = ? WHERE id = ?");
        $stmt->execute([$new_status, $task_id]);
        $success = $new_status ? "تم تحديد المهمة كمكتملة." : "تم إلغاء التحديد.";
    } else {
        $error = "غير مسموح لك تعديل هذه المهمة.";
    }
}

// --- حذف مهمة ---
if (isset($_POST['delete_task'])) {
    $task_id = $_POST['task_id'];
    $stmt = $pdo->prepare("DELETE FROM student_tasks WHERE id = ? AND student_id = ?");
    $stmt->execute([$task_id, $student_id]);
    $success = "تم حذف المهمة.";
}

// --- جلب جميع المهام ---
$stmt = $pdo->prepare("SELECT * FROM student_tasks WHERE student_id = ? ORDER BY is_completed, due_date ASC, created_at DESC");
$stmt->execute([$student_id]);
$tasks = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>المهام والملاحظات - اشراق</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
        }

        /* شريط التنقل العلوي */
        .top-nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding: 15px 25px;
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-weight: 700;
            font-size: 20px;
            color: var(--primary);
        }

        .nav-links {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        .nav-link {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 15px;
            border-radius: var(--border-radius);
            color: var(--dark);
            text-decoration: none;
            transition: var(--transition);
        }

        .nav-link:hover {
            background: var(--light);
            color: var(--primary);
        }

        .nav-link.active {
            background: var(--primary);
            color: white;
        }

        /* رأس الصفحة */
        .page-header {
            text-align: center;
            margin-bottom: 30px;
        }

        .page-header h1 {
            font-size: 32px;
            color: var(--primary);
            margin-bottom: 10px;
        }

        .page-header p {
            color: var(--gray);
            font-size: 18px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* نموذج إضافة مهمة */
        .add-task-card {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .card-title {
            font-size: 22px;
            color: var(--primary);
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }

        .form-input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .form-input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        textarea.form-input {
            min-height: 100px;
            resize: vertical;
        }

        .form-actions {
            display: flex;
            gap: 15px;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
            text-decoration: none;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        /* قائمة المهام */
        .tasks-section {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .tasks-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
        }

        .tasks-title {
            font-size: 22px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .tasks-count {
            background: var(--primary);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        .tasks-container {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }

        .task-card {
            padding: 20px;
            border-radius: var(--border-radius);
            border: 1px solid #eee;
            transition: var(--transition);
            position: relative;
            overflow: hidden;
        }

        .task-card::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: var(--primary);
        }

        .task-card.completed {
            background: #e8f5e8;
        }

        .task-card.completed::before {
            background: #28a745;
        }

        .task-card.overdue {
            background: #ffebee;
        }

        .task-card.overdue::before {
            background: var(--error);
        }

        .task-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        .task-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 10px;
        }

        .task-title {
            font-size: 18px;
            color: var(--dark);
            margin-bottom: 5px;
            flex: 1;
        }

        .task-completed .task-title {
            text-decoration: line-through;
            color: var(--gray);
        }

        .task-actions {
            display: flex;
            gap: 10px;
        }

        .task-btn {
            padding: 6px 12px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 14px;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .task-btn.complete {
            background: #28a745;
            color: white;
        }

        .task-btn.complete:hover {
            background: #218838;
        }

        .task-btn.incomplete {
            background: #6c757d;
            color: white;
        }

        .task-btn.incomplete:hover {
            background: #5a6268;
        }

        .task-btn.delete {
            background: var(--error);
            color: white;
        }

        .task-btn.delete:hover {
            background: #c9302c;
        }

        .task-content {
            margin-bottom: 15px;
        }

        .task-description {
            color: var(--gray);
            margin-bottom: 10px;
            line-height: 1.5;
        }

        .task-meta {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 14px;
            color: var(--gray);
        }

        .due-date {
            color: var(--error);
            font-weight: 600;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: var(--gray);
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .top-nav {
                flex-direction: column;
                gap: 15px;
            }
            
            .nav-links {
                width: 100%;
                justify-content: center;
                flex-wrap: wrap;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .task-header {
                flex-direction: column;
                gap: 15px;
            }
            
            .task-actions {
                width: 100%;
                justify-content: flex-end;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- شريط التنقل العلوي -->
        <div class="top-nav">
            <div class="logo">
                <i class="fas fa-graduation-cap"></i>
                <span>اشراق</span>
            </div>
            
            <div class="nav-links">
                <a href="dashboard.php" class="nav-link">
                    <i class="fas fa-home"></i>
                    الرئيسية
                </a>
                <a href="tasks.php" class="nav-link active">
                    <i class="fas fa-tasks"></i>
                    المهام
                </a>
                <a href="profile.php" class="nav-link">
                    <i class="fas fa-user"></i>
                    الملف الشخصي
                </a>
            </div>
        </div>

        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1><i class="fas fa-tasks"></i> المهام والملاحظات</h1>
            <p>نظم مهامك وملاحظاتك الدراسية في مكان واحد</p>
        </div>

        <!-- رسالة الحالة -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <!-- نموذج إضافة مهمة -->
        <div class="add-task-card">
            <h3 class="card-title"><i class="fas fa-plus-circle"></i> إضافة مهمة جديدة</h3>
            
            <form method="POST">
                <input type="hidden" name="add_task" value="1">
                
                <div class="form-group">
                    <label class="form-label">العنوان *</label>
                    <input type="text" name="title" class="form-input" placeholder="مثلاً: دراسة درس النحو" required>
                </div>

                <div class="form-group">
                    <label class="form-label">الوصف (اختياري)</label>
                    <textarea name="description" class="form-input" placeholder="تفاصيل المهمة..."></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">تاريخ اكمال الانجاز(اختياري)</label>
                    <input type="datetime-local" name="due_date" class="form-input">
                </div>

                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> إضافة المهمة
                    </button>
                </div>
            </form>
        </div>

        <!-- قائمة المهام -->
        <div class="tasks-section">
            <div class="tasks-header">
                <h3 class="tasks-title">
                    <i class="fas fa-list-check"></i> المهام الحالية
                    <span class="tasks-count"><?php echo count($tasks); ?></span>
                </h3>
            </div>

            <div class="tasks-container">
                <?php if (count($tasks) > 0): ?>
                    <?php foreach ($tasks as $task): 
                        $now = new DateTime();
                        $due = $task['due_date'] ? new DateTime($task['due_date']) : null;
                        $is_overdue = $due && $now > $due && !$task['is_completed'];
                        $is_completed = $task['is_completed'];
                    ?>
                        <div class="task-card <?php echo $is_completed ? 'completed' : ($is_overdue ? 'overdue' : ''); ?>">
                            <div class="task-header">
                                <div class="task-title <?php echo $is_completed ? 'task-completed' : ''; ?>">
                                    <h4><?php echo htmlspecialchars($task['title']); ?></h4>
                                </div>
                                
                                <div class="task-actions">
                                    <form method="POST">
                                        <input type="hidden" name="toggle_task" value="1">
                                        <input type="hidden" name="task_id" value="<?php echo $task['id']; ?>">
                                        <button type="submit" class="task-btn <?php echo $is_completed ? 'incomplete' : 'complete'; ?>">
                                            <i class="fas <?php echo $is_completed ? 'fa-undo' : 'fa-check'; ?>"></i>
                                            <?php echo $is_completed ? 'إلغاء' : 'مكتمل'; ?>
                                        </button>
                                    </form>
                                    
                                    <form method="POST" onsubmit="return confirm('هل ترغب في حذف هذه المهمة؟');">
                                        <input type="hidden" name="delete_task" value="1">
                                        <input type="hidden" name="task_id" value="<?php echo $task['id']; ?>">
                                        <button type="submit" class="task-btn delete">
                                            <i class="fas fa-trash"></i> حذف
                                        </button>
                                    </form>
                                </div>
                            </div>
                            
                            <div class="task-content">
                                <?php if ($task['description']): ?>
                                    <p class="task-description"><?php echo htmlspecialchars($task['description']); ?></p>
                                <?php endif; ?>
                                
                                <div class="task-meta">
                                    <?php if ($task['due_date']): ?>
                                        <div class="meta-item due-date">
                                            <i class="fas fa-calendar-day"></i>
                                            <span>تاريخ اكمال الانجاز: <?php echo date('Y-m-d H:i', strtotime($task['due_date'])); ?></span>
                                            <?php if ($is_overdue): ?>
                                                <span class="badge" style="background: var(--error); color: white; padding: 2px 8px; border-radius: 10px; font-size: 12px; margin-right: 5px;">⚠️ متأخر</span>
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="meta-item">
                                        <i class="fas fa-clock"></i>
                                        <span>أنشئت في: <?php echo date('Y-m-d H:i', strtotime($task['created_at'])); ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-clipboard-list"></i>
                        <p>لا توجد مهام بعد. أضف أول مهمة!</p>
                        
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- زر العودة -->
        <div style="text-align: center;">
            <a href="dashboard.php" class="btn btn-secondary">
                <i class="fas fa-arrow-right"></i> العودة للرئيسية
            </a>
        </div>
    </div>

    <script>
        // إضافة تأثيرات عند التركيز على الحقول
        const inputs = document.querySelectorAll('.form-input');
        inputs.forEach(input => {
            input.addEventListener('focus', () => {
                input.parentElement.classList.add('focused');
            });
            input.addEventListener('blur', () => {
                input.parentElement.classList.remove('focused');
            });
        });
    </script>
</body>
</html>