<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isTeacher()) {
    header("Location: ../index.php");
    exit();
}

$teacher_id = $_SESSION['user']['id'];
$error = '';
$success = '';

// --- جلب الصفوف ---
$stmt = $pdo->prepare("SELECT id, name FROM classes WHERE teacher_id = ?");
$stmt->execute([$teacher_id]);
$classes = $stmt->fetchAll();

if (count($classes) == 0) {
    $error = "لم تُنشئ أي صفوف بعد. يرجى إنشاء صف أولًا.";
}

// --- مسار التحميل الآمن ---
$uploadDir = '../assets/uploads/homeworks/';
$fullUploadPath = __DIR__ . '/../../' . ltrim($uploadDir, '../'); // تحويل المسار النسبي إلى مطلق

// التحقق من وجود المجلد وإنشائه إن لم يكن موجودًا
if (!is_dir($fullUploadPath)) {
    if (!mkdir($fullUploadPath, 0755, true)) {
        $error = "فشل إنشاء مجلد التحميل. يرجى التواصل مع المسؤول.";
    }
}

// --- معالجة الحفظ ---
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $class_id = $_POST['class_id'];
    $due_date = !empty($_POST['due_date']) ? $_POST['due_date'] : null;
    $file_path = null;

    // التحقق من الحقول المطلوبة
    if (empty($title)) {
        $error = "عنوان الواجب مطلوب.";
    } elseif (empty($class_id)) {
        $error = "يرجى اختيار الصف.";
    } else {
        // رفع ملف تعليمات (اختياري)
        if (!empty($_FILES['file']['name'])) {
            $ext = strtolower(pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION));
            $allowed = ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'];

            if (!in_array($ext, $allowed)) {
                $error = "نوع الملف غير مسموح به. فقط PDF، صور، أو Word.";
            } elseif ($_FILES['file']['size'] > 5 * 1024 * 1024) { // 5MB
                $error = "حجم الملف أكبر من الحد المسموح به (5MB).";
            } else {
                $filename = 'homeworks/' . uniqid() . '.' . $ext;
                $path = $fullUploadPath . '/' . basename($filename); // المسار المطلق

                if (move_uploaded_file($_FILES['file']['tmp_name'], $path)) {
                    $file_path = $filename; // نحفظ المسار النسبي للقاعدة
                } else {
                    $error = "فشل رفع ملف التعليمات. حاول مرة أخرى.";
                }
            }
        }

        // لا يوجد خطأ → حفظ الواجب
        if (!$error) {
            try {
                $stmt = $pdo->prepare("INSERT INTO homeworks (teacher_id, class_id, title, description, file_path, due_date) 
                                       VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$teacher_id, $class_id, $title, $description, $file_path, $due_date]);
                $success = "تم إنشاء الواجب بنجاح!";
            } catch (Exception $e) {
                $error = "حدث خطأ أثناء الحفظ. حاول لاحقًا.";
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إنشاء واجب جديد - منصة التعليم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            flex: 1;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* نموذج إنشاء الواجب */
        .form-container {
            background: white;
            padding: 30px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .form-title {
            font-size: 24px;
            color: var(--primary);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }

        .form-input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .form-input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .form-select {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            background: white;
            transition: var(--transition);
        }

        .form-select:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .form-textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            min-height: 120px;
            resize: vertical;
            transition: var(--transition);
        }

        .form-textarea:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .file-input {
            width: 100%;
            padding: 10px;
            border: 2px dashed #ddd;
            border-radius: var(--border-radius);
            background: #f9f9f9;
            transition: var(--transition;
        }

        .file-input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .file-hint {
            color: var(--gray);
            font-size: 14px;
            margin-top: 5px;
            display: block;
        }

        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
            flex-wrap: wrap;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        /* نجاح الإنشاء */
        .success-section {
            background: white;
            padding: 40px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            text-align: center;
            margin-bottom: 30px;
        }

        .success-icon {
            font-size: 64px;
            color: var(--success);
            margin-bottom: 20px;
        }

        .success-message {
            font-size: 20px;
            color: var(--dark);
            margin-bottom: 25px;
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .form-actions {
                flex-direction: column;
            }
            
            .btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/header.php'; ?>

    <div class="container">
        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1>
                <i class="fas fa-plus-circle"></i>
                إنشاء واجب جديد
            </h1>
            <p>أضف واجباً جديداً للطلاب ومتابعة تقدمهم</p>
        </div>

        <!-- رسالة الحالة -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <!-- عرض نجاح الإنشاء -->
            <div class="success-section">
                <div class="success-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <h2 class="success-message">تم إنشاء الواجب بنجاح!</h2>
                <p style="color: var(--gray); margin-bottom: 25px;">يمكنك الآن متابعة تسليمات الطلاب لهذا الواجب</p>
                <div class="form-actions">
                    <a href="homeworks.php" class="btn btn-primary">
                        <i class="fas fa-tasks"></i>
                        عرض جميع الواجبات
                    </a>
                    <a href="add_homework.php" class="btn btn-secondary">
                        <i class="fas fa-plus"></i>
                        إنشاء واجب جديد
                    </a>
                </div>
            </div>
            <?php include '../includes/footer.php'; exit(); ?>
        <?php endif; ?>

        <!-- نموذج إنشاء الواجب -->
        <div class="form-container">
            <h2 class="form-title">
                <i class="fas fa-book"></i>
                معلومات الواجب
            </h2>
            
            <form method="POST" enctype="multipart/form-data">
                <div class="form-group">
                    <label class="form-label">عنوان الواجب *</label>
                    <input type="text" name="title" placeholder="مثلاً: واجب درس النحو" 
                           value="<?php echo htmlspecialchars($_POST['title'] ?? ''); ?>"
                           class="form-input" required>
                </div>

                <div class="form-group">
                    <label class="form-label">اختر الصف *</label>
                    <select name="class_id" class="form-select" required>
                        <option value="">اختر الصف</option>
                        <?php foreach ($classes as $c): ?>
                            <option value="<?php echo $c['id']; ?>" 
                                    <?php echo (isset($_POST['class_id']) && $_POST['class_id'] == $c['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($c['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label">وصف او ملحوظة للطالب خاص بالواجب (اختياري)</label>
                    <textarea name="description" rows="4" class="form-textarea"
                              placeholder="تفاصيل الواجب، التعليمات، المتطلبات..."><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">رفع ملف الخاص بالواجب (اختياري)</label>
                    <input type="file" name="file" accept=".pdf,.jpg,.jpeg,.png,.doc,.docx" class="file-input">
                    <span class="file-hint">الملفات المسموحة: PDF، صور، Word - الحد الأقصى: 5MB</span>
                </div>

                <div class="form-group">
                    <label class="form-label">تاريخ تسلم الواجب (اختياري)</label>
                    <input type="datetime-local" name="due_date" 
                           value="<?php echo htmlspecialchars($_POST['due_date'] ?? ''); ?>"
                           class="form-input">
                </div>

                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        إنشاء الواجب
                    </button>
                    <a href="homeworks.php" class="btn btn-secondary">
                        <i class="fas fa-times"></i>
                        إلغاء
                    </a>
                </div>
            </form>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>

    <script>
    // تأثيرات عند التركيز على الحقول
    const inputs = document.querySelectorAll('.form-input, .form-select, .form-textarea, .file-input');
    inputs.forEach(input => {
        input.addEventListener('focus', () => {
            input.parentElement.classList.add('focused');
        });
        input.addEventListener('blur', () => {
            input.parentElement.classList.remove('focused');
        });
    });

    // التحقق من حجم الملف قبل الرفع
    document.querySelector('input[name="file"]').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            // التحقق من حجم الملف (5MB كحد أقصى)
            if (file.size > 5 * 1024 * 1024) {
                alert('حجم الملف يجب أن يكون أقل من 5MB');
                this.value = '';
            }
            
            // التحقق من نوع الملف
            const validTypes = ['application/pdf', 'image/jpeg', 'image/png', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
            if (!validTypes.includes(file.type)) {
                alert('نوع الملف غير مسموح. الرجاء استخدام ملف من نوع PDF، صورة، أو Word');
                this.value = '';
            }
        }
    });

    // إضافة علامة النجمة للحقول الإلزامية
    document.querySelectorAll('.form-label').forEach(label => {
        if (label.textContent.includes('*')) {
            label.innerHTML = label.innerHTML.replace('*', '<span style="color: var(--error);">*</span>');
        }
    });
    </script>
</body>
</html>