<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isTeacher()) {
    header("Location: ../index.php");
    exit();
}

$teacher_id = $_SESSION['user']['id'];
$error = '';
$success = '';
$lesson = null;
$is_edit = false;

// --- حذف درس ---
if (isset($_POST['delete_lesson'])) {
    $lesson_id = $_POST['lesson_id'];

    $stmt = $pdo->prepare("SELECT id FROM lessons WHERE id = ? AND teacher_id = ?");
    $stmt->execute([$lesson_id, $teacher_id]);

    if ($stmt->rowCount() > 0) {
        $stmt = $pdo->prepare("DELETE FROM lessons WHERE id = ?");
        $stmt->execute([$lesson_id]);
        $success = "تم حذف الدرس بنجاح!";
    } else {
        $error = "غير مسموح لك حذف هذا الدرس.";
    }
}

// --- إضافة أو تعديل درس ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && !isset($_POST['delete_lesson'])) {
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $zoom_link = trim($_POST['zoom_link']);
    $recorded_link = trim($_POST['recorded_link'] ?? '');
    $class_ids = isset($_POST['class_ids']) ? $_POST['class_ids'] : []; // Array now
    $start_time = !empty($_POST['start_time']) ? $_POST['start_time'] : null;
    $lesson_type = $_POST['lesson_type'] ?? 'live';

    if (empty($title) && empty($description) && empty($zoom_link) && empty($recorded_link)) {
        $error = "يرجى تعبئة حقل واحد على الأقل.";
    } else {
        if ($lesson_type == 'live' && empty($zoom_link)) {
            $error = "المحاضرة المباشرة تتطلب رابط المنصة المعرض عليها.";
        } elseif ($lesson_type == 'recorded' && empty($recorded_link)) {
            $error = "المحاضرة المسجلة تتطلب رابط التسجيل.";
        }
    }

    if (!$error) {
        try {
            // تحويل مصفوفة class_ids إلى نص مفصول بفواصل
            $class_ids_string = !empty($class_ids) ? implode(',', $class_ids) : null;

            if (isset($_POST['edit_id'])) {
                $stmt = $pdo->prepare("UPDATE lessons SET 
                    title = ?, description = ?, zoom_link = ?, recorded_link = ?, 
                    class_ids = ?, start_time = ?, lesson_type = ?
                    WHERE id = ? AND teacher_id = ?");
                $stmt->execute([
                    $title, $description, $zoom_link, $recorded_link,
                    $class_ids_string, $start_time, $lesson_type,
                    $_POST['edit_id'], $teacher_id
                ]);
                $success = "تم تحديث الدرس بنجاح!";
            } else {
                $stmt = $pdo->prepare("INSERT INTO lessons 
                    (title, description, zoom_link, recorded_link, class_ids, teacher_id, start_time, lesson_type)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $title, $description, $zoom_link, $recorded_link,
                    $class_ids_string, $teacher_id, $start_time, $lesson_type
                ]);
                $success = "تم إضافة الدرس بنجاح!";
            }
        } catch (Exception $e) {
            $error = "حدث خطأ أثناء الحفظ: " . $e->getMessage();
        }
    }
}

// --- التحقق من تعديل درس ---
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $lesson_id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM lessons WHERE id = ? AND teacher_id = ?");
    $stmt->execute([$lesson_id, $teacher_id]);
    $lesson = $stmt->fetch();

    if ($lesson) {
        $is_edit = true;
        // تحويل class_ids من نص إلى مصفوفة لاستخدامها في الواجهة
        $lesson['class_ids_array'] = !empty($lesson['class_ids']) ? explode(',', $lesson['class_ids']) : [];
    } else {
        $error = "الدرس غير موجود أو ليس لك.";
    }
}

// --- جلب الصفوف ---
$classes = $pdo->prepare("SELECT id, name FROM classes WHERE teacher_id = ? ORDER BY name");
$classes->execute([$teacher_id]);
$classes_data = $classes->fetchAll();

// --- جلب جميع الدروس ---
$stmt = $pdo->prepare("SELECT l.*, c.name as class_name FROM lessons l 
                       LEFT JOIN classes c ON FIND_IN_SET(c.id, l.class_ids) > 0 
                       WHERE l.teacher_id = ? 
                       ORDER BY l.start_time DESC, l.created_at DESC");
$stmt->execute([$teacher_id]);
$lessons_raw = $stmt->fetchAll();

// إعادة تجميع الصفوف لكل درس
$lessons = [];
foreach ($lessons_raw as $row) {
    $id = $row['id'];
    if (!isset($lessons[$id])) {
        $lessons[$id] = $row;
        $lessons[$id]['class_names'] = [];
    }
    if ($row['class_name']) {
        $lessons[$id]['class_names'][] = $row['class_name'];
    }
}
$lessons = array_values($lessons); // إعادة ترقيم المفاتيح
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $is_edit ? 'تعديل درس' : 'إضافة درس'; ?> - منصة التعليم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
            flex: 1;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* نموذج إضافة/تعديل درس */
        .form-container {
            background: white;
            padding: 30px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .form-title {
            font-size: 24px;
            color: var(--primary);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        @media (max-width: 768px) {
            .form-grid {
                grid-template-columns: 1fr;
            }
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }

        .form-input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .form-input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        textarea.form-input {
            min-height: 120px;
            resize: vertical;
        }

        .checkbox-group {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 10px;
        }

        .checkbox-item {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 12px;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            transition: var(--transition);
        }

        .checkbox-item:hover {
            background: #e9ecef;
        }

        .checkbox-item input[type="checkbox"] {
            transform: scale(1.2);
        }

        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 25px;
            flex-wrap: wrap;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        .btn-danger {
            background: var(--error);
            color: white;
        }

        .btn-danger:hover {
            background: #c9302c;
            transform: translateY(-2px);
        }

        /* شريط البحث المحسن */
        .search-section {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 20px;
        }

        .search-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
            flex-wrap: wrap;
            gap: 15px;
        }

        .search-title {
            font-size: 20px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .results-count {
            background: var(--primary);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        .search-box {
            position: relative;
        }

        .search-box input {
            width: 100%;
            padding: 12px 45px 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .search-box input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray);
        }

        /* جدول الدروس */
        .lessons-section {
            background: white;
            padding: 30px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
            overflow-x: auto;
        }

        .section-title {
            font-size: 24px;
            color: var(--dark);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .lessons-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .lessons-table th {
            background: var(--primary);
            color: white;
            padding: 15px;
            text-align: right;
            font-weight: 600;
            position: sticky;
            top: 0;
        }

        .lessons-table td {
            padding: 15px;
            border-bottom: 1px solid #eee;
        }

        .lessons-table tr:hover {
            background: #f8f9fa;
        }

        .lesson-type-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        .lesson-type-live {
            background: rgba(67, 97, 238, 0.1);
            color: var(--primary);
        }

        .lesson-type-recorded {
            background: rgba(114, 9, 183, 0.1);
            color: var(--secondary);
        }

        .action-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: var(--gray);
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        .no-results {
            text-align: center;
            padding: 30px;
            color: var(--gray);
            display: none;
        }

        .no-results i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .lessons-table {
                display: block;
                overflow-x: auto;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .search-header {
                flex-direction: column;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/header.php'; ?>

    <div class="container">
        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1>
                <i class="fas fa-chalkboard-teacher"></i>
                <?php echo $is_edit ? 'تعديل درس' : 'إضافة درس جديد'; ?>
            </h1>
        </div>

        <!-- رسالة الحالة -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <!-- نموذج إضافة / تعديل درس -->
        <div class="form-container">
            <h2 class="form-title">
                <i class="fas fa-book"></i>
                <?php echo $is_edit ? 'تعديل الدرس' : 'معلومات الدرس الجديد'; ?>
            </h2>
            
            <form method="POST">
                <?php if ($is_edit): ?>
                    <input type="hidden" name="edit_id" value="<?php echo $lesson['id']; ?>">
                <?php endif; ?>

                <div class="form-grid">
                    <div class="form-group">
                        <label class="form-label">عنوان الدرس</label>
                        <input type="text" name="title" class="form-input" 
                               value="<?php echo htmlspecialchars($lesson['title'] ?? ''); ?>" 
                               placeholder="أدخل عنوان الدرس">
                    </div>

                    <div class="form-group">
                        <label class="form-label">نوع الدرس</label>
                        <select name="lesson_type" id="lesson_type" class="form-input">
                            <option value="live" <?php echo (isset($lesson) && $lesson['lesson_type'] == 'live') ? 'selected' : ''; ?>>محاضرة مباشرة (المنصة المعرض عليها)</option>
                            <option value="recorded" <?php echo (isset($lesson) && $lesson['lesson_type'] == 'recorded') ? 'selected' : ''; ?>>محاضرة مسجلة</option>
                        </select>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">وصف الدرس</label>
                    <textarea name="description" class="form-input" placeholder="أدخل وصف الدرس..."><?php echo htmlspecialchars($lesson['description'] ?? ''); ?></textarea>
                </div>

                <div class="form-grid">
                    <div id="zoom_field" class="form-group" style="<?php echo (isset($lesson) && $lesson['lesson_type'] == 'recorded') ? 'display:none;' : ''; ?>">
                        <label class="form-label">رابط المنصة المعرض عليها</label>
                        <input type="url" name="zoom_link" class="form-input" 
                               value="<?php echo htmlspecialchars($lesson['zoom_link'] ?? ''); ?>" 
                               placeholder="https://zoom.us/j/...  ">
                    </div>

                    <div id="recorded_field" class="form-group" style="<?php echo (isset($lesson) && $lesson['lesson_type'] == 'live') ? 'display:none;' : ''; ?>">
                        <label class="form-label">رابط المحاضرة المسجلة</label>
                        <input type="url" name="recorded_link" class="form-input" 
                               value="<?php echo htmlspecialchars($lesson['recorded_link'] ?? ''); ?>" 
                               placeholder="https://youtube.com/...  ">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">اختر الصفوف (يمكن اختيار أكثر من صف)</label>
                    <div class="checkbox-group">
                        <?php foreach ($classes_data as $c): ?>
                            <label class="checkbox-item">
                                <input type="checkbox" name="class_ids[]" value="<?php echo $c['id']; ?>"
                                    <?php 
                                    if ($is_edit && in_array($c['id'], $lesson['class_ids_array'] ?? [])) {
                                        echo 'checked';
                                    }
                                    ?>>
                                <?php echo htmlspecialchars($c['name']); ?>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">وقت البدء</label>
                    <input type="datetime-local" name="start_time" class="form-input"
                           value="<?php echo !empty($lesson['start_time']) ? date('Y-m-d\TH:i', strtotime($lesson['start_time'])) : ''; ?>">
                </div>

                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        <?php echo $is_edit ? 'تحديث الدرس' : 'إضافة الدرس'; ?>
                    </button>
                    
                    <?php if ($is_edit): ?>
                        <a href="add_lesson.php" class="btn btn-secondary">
                            <i class="fas fa-plus"></i> إضافة جديد
                        </a>
                    <?php endif; ?>
                    
                    <a href="dashboard.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-right"></i> العودة للوحة التحكم
                    </a>
                </div>
            </form>
        </div>

        <!-- شريط البحث المحسن -->
        <div class="search-section">
            <div class="search-header">
                <h3 class="search-title">
                    <i class="fas fa-search"></i>
                    البحث في الدروس
                </h3>
                <span class="results-count" id="resultsCount"><?php echo count($lessons); ?> نتيجة</span>
            </div>
            
            <div class="search-box">
                <input type="text" id="searchLessons" placeholder="ابحث في العنوان، الصف، النوع، أو وقت البدء...">
                <i class="fas fa-search"></i>
            </div>
        </div>

        <!-- عرض الدروس الحالية -->
        <div class="lessons-section">
            <h2 class="section-title">
                <i class="fas fa-list"></i>
                الدروس المضافة
            </h2>

            <?php if (count($lessons) > 0): ?>
                <table class="lessons-table" id="lessonsTable">
                    <thead>
                        <tr>
                            <th>العنوان</th>
                            <th>الصفوف</th>
                            <th>النوع</th>
                            <th>وقت البدء</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($lessons as $l): ?>
                            <?php
                            $class_display = !empty($l['class_names']) ? implode('، ', $l['class_names']) : 'جميع الصفوف';
                            ?>
                            <tr class="lesson-row" 
                                data-title="<?php echo htmlspecialchars(strtolower($l['title'] ?: '')); ?>" 
                                data-class="<?php echo htmlspecialchars(strtolower($class_display)); ?>"
                                data-type="<?php echo $l['lesson_type'] == 'live' ? 'مباشر' : 'مسجل'; ?>"
                                data-time="<?php echo $l['start_time'] ? date('Y-m-d H:i', strtotime($l['start_time'])) : 'غير محدد'; ?>">
                                <td>
                                    <strong><?php echo htmlspecialchars($l['title'] ?: 'بدون عنوان'); ?></strong>
                                    <?php if ($l['description']): ?>
                                        <p style="color: var(--gray); font-size: 14px; margin-top: 5px;">
                                            <?php echo mb_substr(htmlspecialchars($l['description']), 0, 50) . '...'; ?>
                                        </p>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span style="color: var(--primary); font-weight: 600;">
                                        <?php echo htmlspecialchars($class_display); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="lesson-type-badge <?php echo $l['lesson_type'] == 'live' ? 'lesson-type-live' : 'lesson-type-recorded'; ?>">
                                        <?php echo $l['lesson_type'] == 'live' ? 'مباشر' : 'مسجل'; ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo $l['start_time'] ? date('Y-m-d H:i', strtotime($l['start_time'])) : 'غير محدد'; ?>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="add_lesson.php?edit=<?php echo $l['id']; ?>" class="btn" style="background: var(--warning); color: white; padding: 8px 15px; font-size: 14px;">
                                            <i class="fas fa-edit"></i> تعديل
                                        </a>
                                        
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="delete_lesson" value="1">
                                            <input type="hidden" name="lesson_id" value="<?php echo $l['id']; ?>">
                                            <button type="submit" class="btn btn-danger" style="padding: 8px 15px; font-size: 14px;"
                                                    onclick="return confirm('هل أنت متأكد من حذف هذا الدرس؟\n\nملاحظة: لا يمكن التراجع.');">
                                                <i class="fas fa-trash"></i> حذف
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <div class="no-results" id="noResults">
                    <i class="fas fa-search"></i>
                    <p>لا توجد نتائج تطابق بحثك</p>
                    <p>حاول استخدام كلمات بحث مختلفة</p>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-book-open"></i>
                    <p>لم تُضف أي دروس بعد</p>
                    <p>استخدم النموذج أعلاه لإضافة أول درس</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>

    <!-- JavaScript: تغيير الحقول حسب نوع الدرس -->
    <script>
    document.getElementById('lesson_type').addEventListener('change', function () {
        const type = this.value;
        document.getElementById('zoom_field').style.display = type === 'live' ? 'block' : 'none';
        document.getElementById('recorded_field').style.display = type === 'recorded' ? 'block' : 'none';
    });

    // تأثيرات عند التركيز على الحقول
    const inputs = document.querySelectorAll('.form-input');
    inputs.forEach(input => {
        input.addEventListener('focus', () => {
            input.parentElement.classList.add('focused');
        });
        input.addEventListener('blur', () => {
            input.parentElement.classList.remove('focused');
        });
    });

    // البحث التلقائي في جميع حقول الجدول
    document.getElementById('searchLessons').addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase().trim();
        const rows = document.querySelectorAll('.lesson-row');
        let visibleCount = 0;

        rows.forEach(row => {
            const title = row.dataset.title || '';
            const classText = row.dataset.class || '';
            const type = row.dataset.type || '';
            const time = row.dataset.time || '';
            
            if (title.includes(searchTerm) || 
                classText.includes(searchTerm) || 
                type.includes(searchTerm) || 
                time.includes(searchTerm)) {
                row.style.display = '';
                visibleCount++;
            } else {
                row.style.display = 'none';
            }
        });

        // تحديث عدد النتائج
        document.getElementById('resultsCount').textContent = visibleCount + ' نتيجة';
        
        // إظهار/إخفاء رسالة عدم وجود نتائج
        const noResults = document.getElementById('noResults');
        if (visibleCount === 0 && rows.length > 0) {
            noResults.style.display = 'block';
        } else {
            noResults.style.display = 'none';
        }
    });

    // تفعيل البحث عند تحميل الصفحة إذا كان هناك نص في شريط البحث
    document.addEventListener('DOMContentLoaded', function() {
        const searchInput = document.getElementById('searchLessons');
        if (searchInput.value) {
            searchInput.dispatchEvent(new Event('input'));
        }
    });
    </script>
</body>
</html>