<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isTeacher()) {
    header("Location: ../index.php");
    exit();
}

$teacher_id = $_SESSION['user']['id'];
$student_id = $_GET['id'] ?? null;

// --- جلب بيانات الطالب ---
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

if (!$student) {
    die("<h3 style='text-align: center; color: red; margin-top: 50px;'>الطالب غير موجود.</h3>");
}

$error = '';
$success = '';

// --- معالجة الحفظ ---
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $name = trim($_POST['name']);
    $email = trim($_POST['email']);
    $phone = trim($_POST['phone']);
    $class_id = $_POST['class_id'] ?? null; // يمكن أن يكون NULL (لا صف)

    // التحقق من البريد (إذا تغير)
    if ($email !== $student['email']) {
        $check = $pdo->prepare("SELECT id FROM students WHERE email = ? AND id != ?");
        $check->execute([$email, $student_id]);
        if ($check->rowCount() > 0) {
            $error = "هذا البريد الإلكتروني مستخدم من قبل طالب آخر.";
        }
    }

    if (!$error) {
        $image_url = $student['image_url']; // افتراضي: الحفاظ على الصورة القديمة

        // رفع صورة جديدة إن وُجدت
        if (!empty($_FILES['image']['name'])) {
            $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
            $allowed = ['jpg', 'jpeg', 'png', 'gif'];
            if (in_array(strtolower($ext), $allowed)) {
                $filename = 'uploads/' . uniqid() . '.' . $ext;
                $path = '../assets/' . $filename;

                if (move_uploaded_file($_FILES['image']['tmp_name'], $path)) {
                    // احذف الصورة القديمة إذا لم تكن الافتراضية
                    if ($image_url != 'uploads/default.png' && file_exists('../assets/' . $image_url)) {
                        unlink('../assets/' . $image_url);
                    }
                    $image_url = $filename;
                } else {
                    $error = "فشل رفع الصورة.";
                }
            } else {
                $error = "نوع الصورة غير مسموح به. فقط JPG، PNG، GIF.";
            }
        }

        if (!$error) {
            try {
                $stmt = $pdo->prepare("UPDATE students SET 
                    name = ?, email = ?, phone = ?, class_id = ?, image_url = ? 
                    WHERE id = ?");
                $stmt->execute([$name, $email, $phone, $class_id, $image_url, $student_id]);

                $success = "تم تحديث بيانات الطالب بنجاح!";
                
                // تحديث البيانات في الذاكرة (إن كانت مستخدمة لاحقًا)
                $student['name'] = $name;
                $student['email'] = $email;
                $student['phone'] = $phone;
                $student['class_id'] = $class_id;
                $student['image_url'] = $image_url;
            } catch (Exception $e) {
                $error = "حدث خطأ أثناء الحفظ: " . $e->getMessage();
            }
        }
    }
}

// --- جلب الصفوف الخاصة بالمُدرِّس ---
$classes = $pdo->prepare("SELECT id, name FROM classes WHERE teacher_id = ? ORDER BY name");
$classes->execute([$teacher_id]);
$classes_data = $classes->fetchAll();
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تعديل بيانات الطالب - منصة التعليم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
            flex: 1;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* محتوى الصفحة */
        .content-wrapper {
            display: flex;
            gap: 30px;
            flex-wrap: wrap;
            align-items: flex-start;
        }

        /* صورة الطالب */
        .student-image-section {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            text-align: center;
            flex: 1;
            min-width: 250px;
        }

        .student-image {
            width: 150px;
            height: 150px;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid var(--primary);
            margin-bottom: 15px;
        }

        .image-note {
            color: var(--gray);
            font-size: 14px;
            margin-top: 10px;
        }

        /* نموذج التعديل */
        .edit-form {
            background: white;
            padding: 30px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            flex: 2;
            min-width: 300px;
        }

        .form-title {
            font-size: 24px;
            color: var(--primary);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }

        .form-input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .form-input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .form-select {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            background: white;
            transition: var(--transition);
        }

        .form-select:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .file-input {
            width: 100%;
            padding: 10px;
            border: 2px dashed #ddd;
            border-radius: var(--border-radius);
            background: #f9f9f9;
            transition: var(--transition);
        }

        .file-input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .file-hint {
            color: var(--gray);
            font-size: 14px;
            margin-top: 5px;
            display: block;
        }

        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
            flex-wrap: wrap;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .content-wrapper {
                flex-direction: column;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/header.php'; ?>

    <div class="container">
        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1>
                <i class="fas fa-user-edit"></i>
                تعديل بيانات الطالب
            </h1>
            <p>تحديث معلومات الطالب الشخصية والصف الدراسي</p>
        </div>

        <!-- رسالة الحالة -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <div class="content-wrapper">
            <!-- صورة الطالب -->
            <div class="student-image-section">
                <h3>
                    <i class="fas fa-camera"></i>
                    الصورة الحالية
                </h3>
                <img src="../assets/<?php echo htmlspecialchars($student['image_url']); ?>" 
                     alt="صورة الطالب" class="student-image">
                <p class="image-note">يمكنك تغيير الصورة من خلال النموذج</p>
            </div>

            <!-- نموذج التعديل -->
            <form method="POST" enctype="multipart/form-data" class="edit-form">
                <h2 class="form-title">
                    <i class="fas fa-user-graduate"></i>
                    المعلومات الشخصية
                </h2>

                <div class="form-group">
                    <label class="form-label">الاسم الكامل *</label>
                    <input type="text" name="name" value="<?php echo htmlspecialchars($student['name']); ?>" 
                           class="form-input" required>
                </div>

                <div class="form-group">
                    <label class="form-label">البريد الإلكتروني *</label>
                    <input type="email" name="email" value="<?php echo htmlspecialchars($student['email']); ?>" 
                           class="form-input" required>
                </div>

                <div class="form-group">
                    <label class="form-label">رقم الهاتف</label>
                    <input type="text" name="phone" value="<?php echo htmlspecialchars($student['phone']); ?>" 
                           class="form-input">
                </div>

                <div class="form-group">
                    <label class="form-label">الصف الدراسي</label>
                    <select name="class_id" class="form-select">
                        <option value="">بدون صف</option>
                        <?php foreach ($classes_data as $c): ?>
                            <option value="<?php echo $c['id']; ?>" <?php echo ($c['id'] == $student['class_id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($c['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label">تغيير الصورة</label>
                    <input type="file" name="image" accept="image/*" class="file-input">
                    <span class="file-hint">الملفات المسموحة: JPG, PNG, GIF - الحد الأقصى: 2MB</span>
                </div>

                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        حفظ التغييرات
                    </button>
                    <a href="manage_students.php" class="btn btn-secondary">
                        <i class="fas fa-times"></i>
                        إلغاء
                    </a>
                </div>
            </form>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>

    <script>
    // تأثيرات عند التركيز على الحقول
    const inputs = document.querySelectorAll('.form-input, .form-select, .file-input');
    inputs.forEach(input => {
        input.addEventListener('focus', () => {
            input.parentElement.classList.add('focused');
        });
        input.addEventListener('blur', () => {
            input.parentElement.classList.remove('focused');
        });
    });

    // التحقق من حجم الصورة قبل الرفع
    document.querySelector('input[name="image"]').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            // التحقق من حجم الملف (2MB كحد أقصى)
            if (file.size > 2 * 1024 * 1024) {
                alert('حجم الصورة يجب أن يكون أقل من 2MB');
                this.value = '';
            }
            
            // التحقق من نوع الملف
            const validTypes = ['image/jpeg', 'image/png', 'image/gif'];
            if (!validTypes.includes(file.type)) {
                alert('نوع الملف غير مسموح. الرجاء استخدام صورة من نوع JPG, PNG, أو GIF');
                this.value = '';
            }
        }
    });
    </script>
</body>
</html>