<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isTeacher()) {
    header("Location: ../index.php");
    exit();
}

$teacher_id = $_SESSION['user']['id'];
$error = '';
$success = '';

// --- جلب الصفوف ---
$stmt = $pdo->prepare("SELECT id, name FROM classes WHERE teacher_id = ?");
$stmt->execute([$teacher_id]);
$classes = $stmt->fetchAll();

// --- جلب الواجبات ---
$stmt = $pdo->prepare("SELECT h.*, c.name as class_name FROM homeworks h
                       JOIN classes c ON h.class_id = c.id
                       WHERE h.teacher_id = ?
                       ORDER BY h.due_date ASC, h.created_at DESC");
$stmt->execute([$teacher_id]);
$homeworks = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>واجباتي - منصة التعليم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
            flex: 1;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* أزرار سريعة */
        .quick-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            flex-wrap: wrap;
            gap: 15px;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        /* بطاقات الواجبات */
        .homework-card {
            background: white;
            padding: 25px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 20px;
            transition: var(--transition);
            border-left: 4px solid var(--primary);
        }

        .homework-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
        }

        .homework-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
            flex-wrap: wrap;
            gap: 15px;
        }

        .homework-title {
            font-size: 20px;
            color: var(--primary);
            margin: 0;
        }

        .class-badge {
            background: var(--secondary);
            color: white;
            padding: 6px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        .homework-details {
            margin-bottom: 20px;
        }

        .detail-item {
            display: flex;
            margin-bottom: 10px;
            align-items: flex-start;
        }

        .detail-icon {
            color: var(--primary);
            margin-left: 10px;
            margin-top: 3px;
            min-width: 20px;
        }

        .detail-content {
            flex: 1;
            color: var(--dark);
        }

        .detail-content strong {
            color: var(--dark);
        }

        .file-link {
            color: var(--primary);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }

        .file-link:hover {
            text-decoration: underline;
        }

        /* إحصائيات التسليم */
        .submission-stats {
            background: #f8f9fa;
            padding: 15px;
            border-radius: var(--border-radius);
            margin: 20px 0;
            border: 1px solid #e9ecef;
        }

        .stats-text {
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 10px;
        }

        .progress-bar {
            height: 10px;
            background: #e9ecef;
            border-radius: 5px;
            overflow: hidden;
        }

        .progress-fill {
            height: 100%;
            background: var(--success);
            border-radius: 5px;
            transition: width 0.5s ease;
        }

        /* أزرار الإجراءات */
        .action-buttons {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }

        .btn-sm {
            padding: 8px 16px;
            font-size: 14px;
        }

        .btn-success {
            background: var(--success);
            color: white;
        }

        .btn-success:hover {
            background: #3db9e3;
            transform: translateY(-2px);
        }

        .btn-danger {
            background: var(--error);
            color: white;
        }

        .btn-danger:hover {
            background: #c9302c;
            transform: translateY(-2px);
        }

        .btn-disabled {
            background: #e9ecef;
            color: #6c757d;
            cursor: not-allowed;
            padding: 8px 16px;
            border-radius: var(--border-radius);
            font-size: 14px;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        /* حالة عدم وجود واجبات */
        .empty-state {
            background: white;
            padding: 40px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            text-align: center;
            color: var(--gray);
        }

        .empty-state i {
            font-size: 64px;
            margin-bottom: 15px;
            color: #dee2e6;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .homework-header {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/header.php'; ?>

    <div class="container">
        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1>
                <i class="fas fa-tasks"></i>
                الواجبات المطلوبه من الطالب
            </h1>
            <p>إدارة الواجبات المدرسية ومتابعة تسليمات الطلاب</p>
        </div>

        <!-- رسالة الحالة -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <!-- أزرار سريعة -->
        <div class="quick-actions">
            <a href="add_homework.php" class="btn btn-primary">
                <i class="fas fa-plus"></i>
                إنشاء واجب جديد
            </a>
        </div>

        <?php if (count($homeworks) == 0): ?>
            <div class="empty-state">
                <i class="fas fa-clipboard-list"></i>
                <p>لم تُنشئ أي واجب بعد</p>
                <p>ابدأ بإنشاء واجبك الأول لمتابعة تقدم طلابك</p>
                <a href="add_homework.php" class="btn btn-primary">
                    <i class="fas fa-plus"></i>
                    إنشاء واجب جديد
                </a>
            </div>
        <?php else: ?>
            <?php foreach ($homeworks as $hw): 
                // جلب عدد التسليمات
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM submissions s 
                                       JOIN students st ON s.student_id = st.id 
                                       WHERE s.homework_id = ? AND st.class_id = ?");
                $stmt->execute([$hw['id'], $hw['class_id']]);
                $submitted_count = $stmt->fetchColumn();

                // جلب عدد الطلاب في الصف
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM students WHERE class_id = ?");
                $stmt->execute([$hw['class_id']]);
                $total_students = $stmt->fetchColumn();
                
                // حساب نسبة التسليم
                $submission_percentage = $total_students > 0 ? ($submitted_count / $total_students) * 100 : 0;
            ?>
                <div class="homework-card">
                    <div class="homework-header">
                        <h3 class="homework-title"><?php echo htmlspecialchars($hw['title']); ?></h3>
                        <span class="class-badge">
                            <i class="fas fa-users"></i>
                            <?php echo htmlspecialchars($hw['class_name']); ?>
                        </span>
                    </div>

                    <div class="homework-details">
                        <?php if ($hw['description']): ?>
                            <div class="detail-item">
                                <span class="detail-icon"><i class="fas fa-align-left"></i></span>
                                <div class="detail-content">
                                    <strong>الوصف:</strong> <?php echo htmlspecialchars($hw['description']); ?>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($hw['file_path']): ?>
                            <div class="detail-item">
                                <span class="detail-icon"><i class="fas fa-file"></i></span>
                                <div class="detail-content">
                                    <strong>ملف التعليمات او الدرس المطلوبة من الطالب:</strong> 
                                    <a href="../assets/uploads/<?php echo $hw['file_path']; ?>" 
                                       target="_blank" 
                                       class="file-link">
                                        <i class="fas fa-download"></i> تحميل الملف
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>

                        <?php if ($hw['due_date']): ?>
                            <div class="detail-item">
                                <span class="detail-icon"><i class="fas fa-clock"></i></span>
                                <div class="detail-content">
                                    <strong>آخر موعد للتسليم:</strong> 
                                    <?php echo date('Y-m-d H:i', strtotime($hw['due_date'])); ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- إحصائيات التسليم -->
                    <div class="submission-stats">
                        <div class="stats-text">
                            <i class="fas fa-chart-bar"></i>
                            تسليمات الطلاب: 
                            <span style="color: var(--success);"><?php echo $submitted_count; ?></span> 
                            من 
                            <span style="color: var(--primary);"><?php echo $total_students; ?></span>
                        </div>
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: <?php echo $submission_percentage; ?>%;"></div>
                        </div>
                    </div>

                    <!-- أزرار الإجراءات -->
                    <div class="action-buttons">
                        <a href="submissions.php?homework_id=<?php echo $hw['id']; ?>" 
                           class="btn btn-success btn-sm">
                           <i class="fas fa-eye"></i>
                           عرض التسليمات (<?php echo $submitted_count; ?>)
                        </a>

                        <!-- حذف الواجب (فقط إذا لم يُسلّم أحد) -->
                        <?php if ($submitted_count == 0): ?>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="delete_homework" value="1">
                                <input type="hidden" name="homework_id" value="<?php echo $hw['id']; ?>">
                                <button type="submit" class="btn btn-danger btn-sm"
                                        onclick="return confirm('هل أنت متأكد من حذف هذا الواجب؟');">
                                    <i class="fas fa-trash"></i>
                                    حذف
                                </button>
                            </form>
                        <?php else: ?>
                            <span class="btn-disabled">
                                <i class="fas fa-info-circle"></i>
                                لا يمكن الحذف (تم التسليم)
                            </span>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>

        <a href="dashboard.php" class="btn btn-secondary">
            <i class="fas fa-arrow-right"></i>
            العودة للوحة التحكم
        </a>
    </div>

    <!-- معالجة الحذف -->
    <?php
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['delete_homework'])) {
        $homework_id = $_POST['homework_id'];
        
        $stmt = $pdo->prepare("SELECT id FROM homeworks WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$homework_id, $teacher_id]);
        if ($stmt->rowCount() > 0) {
            $stmt = $pdo->prepare("DELETE FROM homeworks WHERE id = ?");
            $stmt->execute([$homework_id]);
            echo "<script>location.href='homeworks.php';</script>";
        }
    }
    ?>

    <?php include '../includes/footer.php'; ?>

    <script>
    // تأثيرات تفاعلية للبطاقات
    document.addEventListener('DOMContentLoaded', function() {
        const cards = document.querySelectorAll('.homework-card');
        
        cards.forEach(card => {
            card.addEventListener('mouseenter', function() {
                this.style.boxShadow = '0 15px 35px rgba(0, 0, 0, 0.1)';
            });
            
            card.addEventListener('mouseleave', function() {
                this.style.boxShadow = '0 10px 25px rgba(0, 0, 0, 0.1)';
            });
        });
    });
    </script>
</body>
</html>