<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isTeacher()) {
    header("Location: ../index.php");
    exit();
}

$teacher_id = $_SESSION['user']['id'];
$error = '';
$success = '';

// --- حذف طالب ---
if (isset($_POST['delete_student'])) {
    $student_id = $_POST['student_id'];

    if ($student_id == $_SESSION['user']['id']) {
        $error = "لا يمكنك حذف حسابك.";
    } else {
        $stmt = $pdo->prepare("SELECT id, image_url FROM students WHERE id = ?");
        $stmt->execute([$student_id]);
        $student = $stmt->fetch();

        if ($student) {
            $image_path = '../assets/' . $student['image_url'];
            if ($student['image_url'] != 'uploads/default.png' && file_exists($image_path)) {
                unlink($image_path);
            }

            // حذف الروابط من جدول student_classes أولاً
            $stmt = $pdo->prepare("DELETE FROM student_classes WHERE student_id = ?");
            $stmt->execute([$student_id]);

            // ثم حذف الطالب
            $stmt = $pdo->prepare("DELETE FROM students WHERE id = ?");
            $stmt->execute([$student_id]);
            $success = "تم حذف الطالب بنجاح!";
        } else {
            $error = "الطالب غير موجود.";
        }
    }
}

// --- حظر / إلغاء حظر طالب ---
if (isset($_POST['toggle_ban'])) {
    $student_id = $_POST['student_id'];

    $stmt = $pdo->prepare("SELECT id, status FROM students WHERE id = ?");
    $stmt->execute([$student_id]);
    $student = $stmt->fetch();

    if (!$student) {
        $error = "الطالب غير موجود.";
    } else {
        $new_status = $student['status'] == 'active' ? 'banned' : 'active';
        $stmt = $pdo->prepare("UPDATE students SET status = ? WHERE id = ?");
        $stmt->execute([$new_status, $student_id]);
        $success = $new_status == 'banned' ? "تم حظر الطالب." : "تم إلغاء حظر الطالب.";
    }
}

// --- ربط الطالب بعدة صفوف ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['assign_multiple_classes'])) {
    $student_id = $_POST['student_id'];
    $selected_classes = $_POST['class_ids'] ?? [];

    // التحقق من أن الصفوف المختارة تابعة للمدرس الحالي
    if (!empty($selected_classes)) {
        $placeholders = str_repeat('?,', count($selected_classes) - 1) . '?';
        $stmt = $pdo->prepare("SELECT id FROM classes WHERE id IN ($placeholders) AND teacher_id = ?");
        $params = $selected_classes;
        $params[] = $teacher_id;
        $stmt->execute($params);
        $valid_classes = $stmt->fetchAll(PDO::FETCH_COLUMN);

        if (count($valid_classes) != count($selected_classes)) {
            $error = "بعض الصفوف غير مسموح لك ربط الطالب بها.";
        } else {
            // حذف الروابط القديمة لهذا الطالب
            $stmt = $pdo->prepare("DELETE FROM student_classes WHERE student_id = ?");
            $stmt->execute([$student_id]);

            // إدخال الروابط الجديدة
            $stmt = $pdo->prepare("INSERT INTO student_classes (student_id, class_id) VALUES (?, ?)");
            foreach ($selected_classes as $class_id) {
                $stmt->execute([$student_id, $class_id]);
            }

            $success = "تم ربط الطالب بالصفوف المحددة بنجاح!";
        }
    } else {
        // إذا لم يتم اختيار أي صف، نحذف كل الروابط
        $stmt = $pdo->prepare("DELETE FROM student_classes WHERE student_id = ?");
        $stmt->execute([$student_id]);
        $success = "تم إزالة الطالب من جميع الصفوف.";
    }
}

// --- إزالة الطالب من صف معين (اختياري) ---
if (isset($_POST['remove_from_specific_class'])) {
    $student_id = $_POST['student_id'];
    $class_id = $_POST['class_id'];

    $stmt = $pdo->prepare("SELECT c.id FROM classes c 
                           JOIN student_classes sc ON c.id = sc.class_id 
                           WHERE sc.student_id = ? AND sc.class_id = ? AND c.teacher_id = ?");
    $stmt->execute([$student_id, $class_id, $teacher_id]);

    if ($stmt->rowCount() > 0) {
        $stmt = $pdo->prepare("DELETE FROM student_classes WHERE student_id = ? AND class_id = ?");
        $stmt->execute([$student_id, $class_id]);
        $success = "تم إزالة الطالب من الصف المحدد.";
    } else {
        $error = "غير مسموح لك تعديل هذا الطالب في هذا الصف.";
    }
}

// --- جلب جميع الصفوف الخاصة بالمُدرِّس ---
$stmt = $pdo->prepare("SELECT id, name FROM classes WHERE teacher_id = ? ORDER BY name");
$stmt->execute([$teacher_id]);
$classes = $stmt->fetchAll();

// --- جلب جميع الطلاب في المنصة مع الصفوف المرتبطة بهم ---
$stmt = $pdo->prepare("
    SELECT s.*, 
           GROUP_CONCAT(c.name SEPARATOR ', ') as class_names,
           GROUP_CONCAT(c.id SEPARATOR ',') as class_ids
    FROM students s 
    LEFT JOIN student_classes sc ON s.id = sc.student_id 
    LEFT JOIN classes c ON sc.class_id = c.id 
    GROUP BY s.id 
    ORDER BY s.name
");
$stmt->execute();
$students = $stmt->fetchAll();

// تحويل class_ids إلى مصفوفة لكل طالب لاستخدامها في JavaScript
foreach ($students as &$student) {
    $student['class_ids_array'] = $student['class_ids'] ? explode(',', $student['class_ids']) : [];
}
unset($student); // تحرير المرجع
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الطلاب - منصة التعليم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
            flex: 1;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* شريط البحث والتصفية */
        .search-section {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 20px;
        }

        .search-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
            flex-wrap: wrap;
            gap: 15px;
        }

        .search-title {
            font-size: 20px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .results-count {
            background: var(--primary);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        .search-box {
            position: relative;
            margin-bottom: 15px;
        }

        .search-box input {
            width: 100%;
            padding: 12px 45px 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .search-box input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray);
        }

        .filters {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }

        .filter-label {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }

        .filter-select {
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 14px;
            min-width: 150px;
        }

        /* جدول الطلاب */
        .students-section {
            background: white;
            padding: 30px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
            overflow-x: auto;
        }

        .section-title {
            font-size: 24px;
            color: var(--dark);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .students-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .students-table th {
            background: var(--primary);
            color: white;
            padding: 15px;
            text-align: right;
            font-weight: 600;
            position: sticky;
            top: 0;
        }

        .students-table td {
            padding: 15px;
            border-bottom: 1px solid #eee;
        }

        .students-table tr:hover {
            background: #f8f9fa;
        }

        .student-image {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--primary);
        }

        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        .status-active {
            background: rgba(76, 201, 240, 0.1);
            color: var(--success);
        }

        .status-banned {
            background: rgba(217, 83, 79, 0.1);
            color: var(--error);
        }

        .action-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        .btn-warning {
            background: var(--warning);
            color: black;
        }

        .btn-warning:hover {
            background: #e0a800;
            transform: translateY(-2px);
        }

        .btn-danger {
            background: var(--error);
            color: white;
        }

        .btn-danger:hover {
            background: #c9302c;
            transform: translateY(-2px);
        }

        .btn-success {
            background: #28a745;
            color: white;
        }

        .btn-success:hover {
            background: #218838;
            transform: translateY(-2px);
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: var(--gray);
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        .no-results {
            text-align: center;
            padding: 30px;
            color: var(--gray);
            display: none;
        }

        .no-results i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        /* المودال */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background: white;
            width: 90%;
            max-width: 600px;
            border-radius: var(--border-radius);
            padding: 25px;
            box-shadow: var(--shadow);
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eee;
        }

        .modal-title {
            font-size: 20px;
            color: var(--primary);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .close-modal {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: var(--gray);
        }

        .modal-body {
            margin-bottom: 20px;
        }

        .modal-footer {
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }

        /* نمط checkboxes داخل المودال */
        .checkbox-group {
            margin: 15px 0;
        }

        .checkbox-item {
            display: flex;
            align-items: center;
            margin: 8px 0;
        }

        .checkbox-item input {
            margin-left: 10px;
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .students-table {
                display: block;
                overflow-x: auto;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .search-header {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .filters {
                flex-direction: column;
                width: 100%;
            }
            
            .filter-select {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/header.php'; ?>

    <div class="container">
        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1>
                <i class="fas fa-users"></i>
                إدارة الطلاب
            </h1>
            <p>إدارة جميع طلاب المنصة وربطهم بعدة صفوف</p>
        </div>

        <!-- رسالة الحالة -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <!-- شريط البحث والتصفية -->
        <div class="search-section">
            <div class="search-header">
                <h3 class="search-title">
                    <i class="fas fa-search"></i>
                    البحث والتصفية
                </h3>
                <span class="results-count" id="resultsCount"><?php echo count($students); ?> نتيجة</span>
            </div>
            
            <div class="search-box">
                <input type="text" id="searchStudents" placeholder="ابحث عن طالب (الاسم، البريد، الصف، الهاتف)...">
                <i class="fas fa-search"></i>
            </div>
            
            <div class="filters">
                <div class="filter-group">
                    <span class="filter-label">حسب الصف</span>
                    <select class="filter-select" id="filterClass">
                        <option value="">جميع الصفوف</option>
                        <?php foreach ($classes as $c): ?>
                            <option value="<?php echo $c['id']; ?>"><?php echo htmlspecialchars($c['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <span class="filter-label">حسب الحالة</span>
                    <select class="filter-select" id="filterStatus">
                        <option value="">جميع الحالات</option>
                        <option value="active">نشط</option>
                        <option value="banned">محظور</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <span class="filter-label">حسب الصف المدرسي</span>
                    <select class="filter-select" id="filterClassAssigned">
                        <option value="">الكل</option>
                        <option value="assigned">مرتبط بصف</option>
                        <option value="not_assigned">غير مرتبط</option>
                    </select>
                </div>
            </div>
        </div>

        <!-- جدول الطلاب -->
        <div class="students-section">
            <h2 class="section-title">
                <i class="fas fa-list"></i>
                قائمة الطلاب
            </h2>

            <?php if (count($students) > 0): ?>
                <table class="students-table" id="studentsTable">
                    <thead>
                        <tr>
                            <th>الصورة</th>
                            <th>الاسم</th>
                            <th>البريد الإلكتروني</th>
                            <th>الصفوف</th>
                            <th>الحالة</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($students as $s): ?>
                            <tr class="student-row" 
                                data-id="<?php echo $s['id']; ?>"
                                data-name="<?php echo htmlspecialchars(strtolower($s['name'])); ?>" 
                                data-email="<?php echo htmlspecialchars(strtolower($s['email'])); ?>"
                                data-phone="<?php echo htmlspecialchars(strtolower($s['phone'] ?? '')); ?>"
                                data-class="<?php echo htmlspecialchars(strtolower($s['class_names'] ?? '')); ?>"
                                data-class-ids="<?php echo htmlspecialchars($s['class_ids'] ?? ''); ?>"
                                data-status="<?php echo $s['status']; ?>">
                                <td style="text-align: center;">
                                    <img src="../assets/<?php echo htmlspecialchars($s['image_url']); ?>" 
                                         alt="صورة الطالب" class="student-image">
                                </td>
                                <td>
                                    <strong><?php echo htmlspecialchars($s['name']); ?></strong>
                                </td>
                                <td><?php echo htmlspecialchars($s['email']); ?></td>
                                <td style="text-align: center; color: var(--primary); font-weight: 600;">
                                    <?php echo $s['class_names'] ? htmlspecialchars($s['class_names']) : '<span style="color: var(--gray);">لا يوجد</span>'; ?>
                                </td>
                                <td style="text-align: center;">
                                    <span class="status-badge <?php echo $s['status'] == 'active' ? 'status-active' : 'status-banned'; ?>">
                                        <?php echo $s['status'] == 'active' ? 'نشط' : 'محظور'; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <!-- تعديل -->
                                        <a href="edit_student.php?id=<?php echo $s['id']; ?>" class="btn btn-warning">
                                            <i class="fas fa-edit"></i> تعديل
                                        </a>

                                        <!-- ربط بعدة صفوف -->
                                        <button type="button" 
                                                class="btn btn-primary assignMultipleClassesBtn" 
                                                data-student-id="<?php echo $s['id']; ?>"
                                                data-student-name="<?php echo htmlspecialchars($s['name']); ?>"
                                                data-class-ids='<?php echo json_encode($s['class_ids_array']); ?>'>
                                            <i class="fas fa-link"></i> ربط بصفوف
                                        </button>

                                        <!-- حظر / إلغاء حظر -->
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="toggle_ban" value="1">
                                            <input type="hidden" name="student_id" value="<?php echo $s['id']; ?>">
                                            <button type="submit" class="btn <?php echo $s['status'] == 'active' ? 'btn-danger' : 'btn-success'; ?>"
                                                    onclick="return confirm('<?php echo $s['status'] == 'active' ? 'هل ترغب في حظر هذا الطالب؟' : 'هل ترغب في إلغاء حظر هذا الطالب؟'; ?>');">
                                                <i class="fas <?php echo $s['status'] == 'active' ? 'fa-ban' : 'fa-check'; ?>"></i>
                                                <?php echo $s['status'] == 'active' ? 'حظر' : 'إلغاء الحظر'; ?>
                                            </button>
                                        </form>

                                        <!-- حذف -->
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="delete_student" value="1">
                                            <input type="hidden" name="student_id" value="<?php echo $s['id']; ?>">
                                            <button type="submit" class="btn btn-danger"
                                                    onclick="return confirm('هل أنت متأكد من حذف هذا الطالب نهائياً؟\n\nملاحظة: لا يمكن التراجع.');">
                                                <i class="fas fa-trash"></i> حذف
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <div class="no-results" id="noResults">
                    <i class="fas fa-search"></i>
                    <p>لا توجد نتائج تطابق بحثك</p>
                    <p>حاول استخدام كلمات بحث مختلفة أو تغيير عوامل التصفية</p>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-user-graduate"></i>
                    <p>لا يوجد طلاب في المنصة بعد</p>
                    <p>سيظهر الطلاب هنا عند تسجيلهم في المنصة</p>
                </div>
            <?php endif; ?>
        </div>

        <a href="dashboard.php" class="btn btn-secondary">
            <i class="fas fa-arrow-right"></i> العودة للوحة التحكم
        </a>
    </div>

    <!-- مودال ربط الطالب بعدة صفوف -->
    <div class="modal" id="assignMultipleModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">
                    <i class="fas fa-link"></i>
                    ربط الطالب بعدة صفوف
                </h3>
                <button class="close-modal">&times;</button>
            </div>
            <div class="modal-body">
                <p>الطالب: <strong id="modalStudentNameMultiple"></strong></p>
                <form method="POST" id="assignMultipleForm">
                    <input type="hidden" name="assign_multiple_classes" value="1">
                    <input type="hidden" name="student_id" id="modalStudentIdMultiple">
                    <div class="checkbox-group">
                        <label style="display: block; margin-bottom: 15px; font-weight: 600;">اختر الصفوف المراد ربط الطالب بها:</label>
                        <?php foreach ($classes as $c): ?>
                            <div class="checkbox-item">
                                <input type="checkbox" name="class_ids[]" value="<?php echo $c['id']; ?>" id="class_<?php echo $c['id']; ?>">
                                <label for="class_<?php echo $c['id']; ?>"><?php echo htmlspecialchars($c['name']); ?></label>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" id="cancelAssignMultiple">إلغاء</button>
                <button type="submit" form="assignMultipleForm" class="btn btn-primary">حفظ التغييرات</button>
            </div>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>

    <script>
    document.addEventListener('DOMContentLoaded', function () {
        const modal = document.getElementById('assignMultipleModal');
        const modalName = document.getElementById('modalStudentNameMultiple');
        const modalId = document.getElementById('modalStudentIdMultiple');
        const cancelBtn = document.getElementById('cancelAssignMultiple');
        const closeModalBtn = document.querySelector('.close-modal');
        const assignButtons = document.querySelectorAll('.assignMultipleClassesBtn');
        const searchInput = document.getElementById('searchStudents');
        const filterClass = document.getElementById('filterClass');
        const filterStatus = document.getElementById('filterStatus');
        const filterClassAssigned = document.getElementById('filterClassAssigned');
        const tableRows = document.querySelectorAll('.student-row');
        const noResults = document.getElementById('noResults');
        const resultsCount = document.getElementById('resultsCount');

        // --- فتح مودال الربط بعدة صفوف ---
        assignButtons.forEach(btn => {
            btn.addEventListener('click', function () {
                const studentId = this.getAttribute('data-student-id');
                const studentName = this.getAttribute('data-student-name');
                const classIds = JSON.parse(this.getAttribute('data-class-ids') || '[]');
                
                modalName.textContent = studentName;
                modalId.value = studentId;
                
                // إعادة تعيين جميع checkboxes
                document.querySelectorAll('input[name="class_ids[]"]').forEach(checkbox => {
                    checkbox.checked = false;
                });
                
                // تحديد الصفوف المرتبطة مسبقاً
                classIds.forEach(id => {
                    const checkbox = document.getElementById('class_' + id);
                    if (checkbox) checkbox.checked = true;
                });
                
                modal.style.display = 'flex';
            });
        });

        // --- إغلاق المودال ---
        cancelBtn.addEventListener('click', () => modal.style.display = 'none');
        closeModalBtn.addEventListener('click', () => modal.style.display = 'none');
        modal.addEventListener('click', e => { if (e.target === modal) modal.style.display = 'none'; });

        // --- وظيفة البحث والتصفية ---
        function filterStudents() {
            const searchTerm = searchInput.value.toLowerCase().trim();
            const classFilter = filterClass.value;
            const statusFilter = filterStatus.value;
            const classAssignedFilter = filterClassAssigned.value;
            
            let visibleCount = 0;

            tableRows.forEach(row => {
                const name = row.dataset.name || '';
                const email = row.dataset.email || '';
                const phone = row.dataset.phone || '';
                const className = row.dataset.class || '';
                const classIds = row.dataset.classIds ? row.dataset.classIds.split(',') : [];
                const status = row.dataset.status || '';
                
                const matchesSearch = searchTerm === '' || 
                    name.includes(searchTerm) || 
                    email.includes(searchTerm) || 
                    phone.includes(searchTerm) || 
                    className.includes(searchTerm);
                
                const matchesClass = classFilter === '' || classIds.includes(classFilter);
                
                const matchesStatus = statusFilter === '' || status === statusFilter;
                
                let matchesClassAssigned = true;
                if (classAssignedFilter === 'assigned') {
                    matchesClassAssigned = classIds.length > 0;
                } else if (classAssignedFilter === 'not_assigned') {
                    matchesClassAssigned = classIds.length === 0;
                }
                
                if (matchesSearch && matchesClass && matchesStatus && matchesClassAssigned) {
                    row.style.display = '';
                    visibleCount++;
                } else {
                    row.style.display = 'none';
                }
            });

            resultsCount.textContent = visibleCount + ' نتيجة';
            
            if (visibleCount === 0 && tableRows.length > 0) {
                noResults.style.display = 'block';
            } else {
                noResults.style.display = 'none';
            }
        }

        // --- إضافة مستمعي الأحداث ---
        searchInput.addEventListener('input', filterStudents);
        filterClass.addEventListener('change', filterStudents);
        filterStatus.addEventListener('change', filterStudents);
        filterClassAssigned.addEventListener('change', filterStudents);

        if (searchInput.value || filterClass.value || filterStatus.value || filterClassAssigned.value) {
            filterStudents();
        }
    });
    </script>
</body>
</html>