<?php
require_once '../includes/auth.php';
require_once '../includes/db.php';

if (!isTeacher()) {
    header("Location: ../index.php");
    exit();
}

$teacher_id = $_SESSION['user']['id'];
$error = '';
$success = '';
$material = null;
$is_edit = false;

// --- حذف مادة ---
if (isset($_POST['delete_material'])) {
    $material_id = $_POST['material_id'];

    $stmt = $pdo->prepare("SELECT id FROM materials WHERE id = ? AND uploaded_by = ?");
    $stmt->execute([$material_id, $teacher_id]);

    if ($stmt->rowCount() > 0) {
        $stmt = $pdo->prepare("DELETE FROM materials WHERE id = ?");
        $stmt->execute([$material_id]);
        echo json_encode(['success' => 'تم حذف المادة بنجاح!']);
        exit();
    } else {
        echo json_encode(['error' => 'غير مسموح لك حذف هذه المادة.']);
        exit();
    }
}

// --- جلب الطلاب مع دعم البحث ---
$search_term = $_GET['search'] ?? '';
$search_sql = " WHERE status = 'active' ";
$search_params = [];

if (!empty($search_term)) {
    $search_sql .= " AND (name LIKE ? OR email LIKE ? OR phone LIKE ?) ";
    $search_params = ["%$search_term%", "%$search_term%", "%$search_term%"];
}

$stmt = $pdo->prepare("SELECT id, name, email, phone FROM students " . $search_sql . " ORDER BY name");
$stmt->execute($search_params);
$all_students = $stmt->fetchAll();

// --- جلب الطلاب المرتبطين بمادة معينة (في حالة التعديل) ---
$selected_students = [];
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $material_id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM materials WHERE id = ? AND uploaded_by = ?");
    $stmt->execute([$material_id, $teacher_id]);
    $material = $stmt->fetch();

    if ($material) {
        $is_edit = true;
        $stmt = $pdo->prepare("SELECT student_id FROM material_students WHERE material_id = ?");
        $stmt->execute([$material_id]);
        $selected_students = $stmt->fetchAll(PDO::FETCH_COLUMN);
    } else {
        $error = "المادة غير موجودة أو ليست لك.";
    }
}

// --- إضافة أو تعديل مادة (عبر AJAX) ---
if (isset($_POST['action']) && $_POST['action'] == 'save_material') {
    try {
        $title = trim($_POST['title']);
        $description = trim($_POST['description']);
        $student_ids = isset($_POST['student_ids']) ? $_POST['student_ids'] : [];

        if (empty($title)) {
            throw new Exception("يرجى إدخال عنوان للمادة.");
        }
        if (empty($student_ids)) {
            throw new Exception("يرجى اختيار طالب واحد على الأقل.");
        }

        $file_path = $material['file_path'] ?? null;
        $file_type = $material['file_type'] ?? 'other';

        if (!$is_edit && empty($_POST['existing_file'])) {
            throw new Exception("يرجى اختيار ملف.");
        }

        if (!$is_edit && !empty($_POST['existing_file'])) {
            $file_path = $_POST['existing_file'];
            $file_type = $_POST['file_type'];
        }

        if ($is_edit) {
            $stmt = $pdo->prepare("
                UPDATE materials 
                SET title = ?, description = ?, file_type = ?
                WHERE id = ? AND uploaded_by = ?
            ");
            $stmt->execute([$title, $description, $file_type, $_POST['edit_id'], $teacher_id]);
            $material_id = $_POST['edit_id'];
        } else {
            $stmt = $pdo->prepare("
                INSERT INTO materials (title, description, file_path, file_type, uploaded_by)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([$title, $description, $file_path, $file_type, $teacher_id]);
            $material_id = $pdo->lastInsertId();
        }

        $stmt = $pdo->prepare("DELETE FROM material_students WHERE material_id = ?");
        $stmt->execute([$material_id]);

        foreach ($student_ids as $student_id) {
            $stmt = $pdo->prepare("INSERT INTO material_students (material_id, student_id) VALUES (?, ?)");
            $stmt->execute([$material_id, $student_id]);
        }

        echo json_encode(['success' => $is_edit ? 'تم تحديث المادة بنجاح!' : 'تم رفع المادة بنجاح!']);
        exit();
    } catch (Exception $e) {
        echo json_encode(['error' => $e->getMessage()]);
        exit();
    }
}

// --- جلب المواد ---
$stmt = $pdo->prepare("
    SELECT m.*, GROUP_CONCAT(s.name SEPARATOR '، ') as student_names
    FROM materials m
    LEFT JOIN material_students ms ON m.id = ms.material_id
    LEFT JOIN students s ON ms.student_id = s.id
    WHERE m.uploaded_by = ?
    GROUP BY m.id
    ORDER BY m.uploaded_at DESC
");
$stmt->execute([$teacher_id]);
$materials = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $is_edit ? 'تعديل مادة' : 'رفع مادة تعليمية'; ?> - منصة التعليم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4361ee;
            --primary-dark: #3a56d4;
            --secondary: #7209b7;
            --accent: #f72585;
            --light: #f8f9fa;
            --dark: #2b2d42;
            --gray: #6c757d;
            --success: #4cc9f0;
            --error: #d9534f;
            --warning: #ffc107;
            --border-radius: 12px;
            --shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Cairo', sans-serif;
        }

        body {
            background: #f4f6f9;
            color: var(--dark);
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        /* رأس الصفحة */
        .page-header {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            padding: 25px;
            border-radius: var(--border-radius);
            margin-bottom: 30px;
            box-shadow: var(--shadow);
            text-align: center;
        }

        .page-header h1 {
            font-size: 32px;
            margin-bottom: 10px;
        }

        /* رسائل الحالة */
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 25px;
            display: none;
            align-items: center;
            gap: 15px;
        }

        .alert-error {
            background: #ffebee;
            color: var(--error);
            border: 1px solid #f5c6cb;
        }

        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #c3e6cb;
        }

        /* نموذج الرفع / التعديل */
        .form-container {
            background: white;
            padding: 30px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .form-title {
            font-size: 24px;
            color: var(--primary);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        @media (max-width: 768px) {
            .form-grid {
                grid-template-columns: 1fr;
            }
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }

        .form-input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .form-input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        textarea.form-input {
            min-height: 120px;
            resize: vertical;
        }

        /* شريط التقدم */
        #uploadProgressContainer {
            display: none;
            margin: 20px 0;
        }

        #uploadProgressBar {
            width: 100%;
            height: 10px;
            background: #f1f1f1;
            border-radius: 5px;
            overflow: hidden;
        }

        #uploadProgress {
            height: 100%;
            background: var(--primary);
            width: 0%;
            transition: width 0.3s ease;
        }

        /* الطلاب */
        .students-section {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }

        .students-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            flex-wrap: wrap;
            gap: 15px;
        }

        .search-box {
            position: relative;
            flex: 1;
            min-width: 250px;
        }

        .search-box input {
            width: 100%;
            padding: 12px 45px 12px 15px;
            border: 2px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }

        .search-box input:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray);
        }

        .checkbox-group {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 10px;
            max-height: 300px;
            overflow-y: auto;
            padding: 15px;
            border: 1px solid #eee;
            border-radius: var(--border-radius);
            background: #fafafa;
        }

        .checkbox-item {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 10px 12px;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            transition: var(--transition);
            flex: 1 0 45%;
            min-width: 250px;
            font-size: 14px;
        }

        .checkbox-item:hover {
            background: #e9ecef;
        }

        .checkbox-item input[type="checkbox"] {
            transform: scale(1.2);
        }

        .student-info {
            flex: 1;
        }

        .student-name {
            font-weight: 600;
            color: var(--dark);
        }

        .student-meta {
            color: var(--gray);
            font-size: 12px;
            margin-top: 3px;
        }

        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 25px;
            flex-wrap: wrap;
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: var(--border-radius);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        .btn-danger {
            background: var(--error);
            color: white;
        }

        .btn-danger:hover {
            background: #c9302c;
            transform: translateY(-2px);
        }

        /* شريط البحث */
        .search-section {
            background: white;
            padding: 20px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 20px;
        }

        .search-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
            flex-wrap: wrap;
            gap: 15px;
        }

        .search-title {
            font-size: 20px;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .results-count {
            background: var(--primary);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        /* قائمة المواد */
        .materials-section {
            background: white;
            padding: 30px;
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            margin-bottom: 30px;
            overflow-x: auto;
        }

        .section-title {
            font-size: 24px;
            color: var(--dark);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f1f1;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .materials-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .materials-table th {
            background: var(--primary);
            color: white;
            padding: 15px;
            text-align: right;
            font-weight: 600;
            position: sticky;
            top: 0;
        }

        .materials-table td {
            padding: 15px;
            border-bottom: 1px solid #eee;
        }

        .materials-table tr:hover {
            background: #f8f9fa;
        }

        .file-type-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }

        .file-type-pdf {
            background: rgba(211, 47, 47, 0.1);
            color: #d32f2f;
        }

        .file-type-image {
            background: rgba(76, 201, 240, 0.1);
            color: #4cc9f0;
        }

        .action-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: var(--gray);
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        .empty-state p {
            font-size: 18px;
            margin-bottom: 20px;
        }

        .no-results {
            text-align: center;
            padding: 30px;
            color: var(--gray);
            display: none;
        }

        .no-results i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #ddd;
        }

        /* التكيف مع الشاشات الصغيرة */
        @media (max-width: 768px) {
            .materials-table {
                display: block;
                overflow-x: auto;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .search-header {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .students-header {
                flex-direction: column;
                align-items: stretch;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- رأس الصفحة -->
        <div class="page-header">
            <h1>
                <i class="fas fa-upload"></i>
                <?php echo $is_edit ? 'تعديل مادة' : 'رفع مادة تعليمية جديدة'; ?>
            </h1>
        </div>

        <!-- رسالة الحالة -->
        <div class="alert alert-error" id="errorMessage">
            <i class="fas fa-exclamation-circle"></i>
            <span id="errorText"></span>
        </div>
        
        <div class="alert alert-success" id="successMessage">
            <i class="fas fa-check-circle"></i>
            <span id="successText"></span>
        </div>

        <!-- نموذج الرفع / التعديل -->
        <div class="form-container">
            <h2 class="form-title">
                <i class="fas fa-file"></i>
                <?php echo $is_edit ? 'تعديل المادة' : 'معلومات المادة الجديدة'; ?>
            </h2>
            
            <form id="uploadForm">
                <?php if ($is_edit): ?>
                    <input type="hidden" name="edit_id" value="<?php echo $material['id']; ?>">
                    <input type="hidden" name="existing_file" value="<?php echo $material['file_path']; ?>">
                    <input type="hidden" name="file_type" value="<?php echo $material['file_type']; ?>">
                <?php endif; ?>
                
                <input type="hidden" name="action" value="save_material">

                <div class="form-group">
                    <label class="form-label">عنوان المادة *</label>
                    <input type="text" name="title" class="form-input" required
                           value="<?php echo htmlspecialchars($material['title'] ?? ''); ?>" 
                           placeholder="أدخل عنوان المادة">
                </div>

                <div class="form-group">
                    <label class="form-label">الوصف (اختياري)</label>
                    <textarea name="description" class="form-input" placeholder="أدخل وصفًا للمادة..."><?php echo htmlspecialchars($material['description'] ?? ''); ?></textarea>
                </div>

                <?php if (!$is_edit): ?>
                <div class="form-group">
                    <label class="form-label">اختر الملف *</label>
                    <input type="file" name="file" id="fileInput" accept=".jpg,.jpeg,.png,.gif,.pdf" required>
                    <small style="color: #666; display: block; margin-top: 5px;">
                        الأنواع المدعومة: صور (JPG, PNG, GIF) و PDF - الحجم الأقصى: 50 ميجابايت
                    </small>
                    
                    <!-- شريط التقدم -->
                    <div id="uploadProgressContainer">
                        <div id="uploadProgressBar">
                            <div id="uploadProgress"></div>
                        </div>
                        <p id="uploadProgressText" style="text-align: center; margin-top: 5px; font-size: 14px;">0%</p>
                    </div>
                </div>
                <?php endif; ?>

                <!-- قسم الطلاب مع البحث -->
                <div class="students-section">
                    <div class="students-header">
                        <h3 class="form-label">اختر الطلاب الذين سيتمكنون من رؤية هذه المادة *</h3>
                        
                        <div class="search-box">
                            <input type="text" id="studentSearch" placeholder="ابحث عن طالب بالاسم، البريد، أو الهاتف...">
                            <i class="fas fa-search"></i>
                        </div>
                    </div>
                    
                    <div class="checkbox-group" id="studentsContainer">
                        <?php if (empty($all_students)): ?>
                            <p style="color: var(--gray); width: 100%; text-align: center;">لا يوجد طلاب مطابقين للبحث.</p>
                        <?php else: ?>
                            <?php foreach ($all_students as $student): ?>
                                <label class="checkbox-item">
                                    <input type="checkbox" name="student_ids[]" value="<?php echo $student['id']; ?>"
                                        <?php 
                                        if (in_array($student['id'], $selected_students)) {
                                            echo 'checked';
                                        }
                                        ?>>
                                    <div class="student-info">
                                        <div class="student-name"><?php echo htmlspecialchars($student['name']); ?></div>
                                        <div class="student-meta">
                                            <?php echo htmlspecialchars($student['email']); ?> | <?php echo htmlspecialchars($student['phone'] ?? '---'); ?>
                                        </div>
                                    </div>
                                </label>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="form-actions">
                    <button type="submit" class="btn btn-primary" id="submitBtn">
                        <i class="fas fa-save"></i>
                        <?php echo $is_edit ? 'تحديث المادة' : 'رفع المادة'; ?>
                    </button>
                    
                    <?php if ($is_edit): ?>
                        <a href="upload_material.php" class="btn btn-secondary">
                            <i class="fas fa-plus"></i> رفع جديد
                        </a>
                    <?php endif; ?>
                    
                    <a href="dashboard.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-right"></i> العودة للوحة التحكم
                    </a>
                </div>
            </form>
        </div>

        <!-- شريط البحث -->
        <div class="search-section">
            <div class="search-header">
                <h3 class="search-title">
                    <i class="fas fa-search"></i>
                    البحث في المواد
                </h3>
                <span class="results-count" id="resultsCount"><?php echo count($materials); ?> نتيجة</span>
            </div>
            
            <div class="search-box">
                <input type="text" id="searchMaterials" placeholder="ابحث في العنوان، الطلاب، أو نوع الملف...">
                <i class="fas fa-search"></i>
            </div>
        </div>

        <!-- عرض المواد الحالية -->
        <div class="materials-section">
            <h2 class="section-title">
                <i class="fas fa-list"></i>
                المواد المضافة
            </h2>

            <?php if (count($materials) > 0): ?>
                <table class="materials-table" id="materialsTable">
                    <thead>
                        <tr>
                            <th>العنوان</th>
                            <th>الطلاب</th>
                            <th>النوع</th>
                            <th>تاريخ الرفع</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($materials as $m): ?>
                            <tr class="material-row" 
                                data-title="<?php echo htmlspecialchars(strtolower($m['title'])); ?>" 
                                data-students="<?php echo htmlspecialchars(strtolower($m['student_names'] ?? '')); ?>"
                                data-type="<?php echo $m['file_type']; ?>">
                                <td>
                                    <strong><?php echo htmlspecialchars($m['title']); ?></strong>
                                    <?php if ($m['description']): ?>
                                        <p style="color: var(--gray); font-size: 14px; margin-top: 5px;">
                                            <?php echo mb_substr(htmlspecialchars($m['description']), 0, 50) . '...'; ?>
                                        </p>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span style="color: var(--primary); font-weight: 600;">
                                        <?php echo htmlspecialchars($m['student_names'] ?: 'لم يتم التعيين'); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="file-type-badge file-type-<?php echo $m['file_type']; ?>">
                                        <?php 
                                        if ($m['file_type'] == 'pdf') echo 'PDF';
                                        elseif ($m['file_type'] == 'image') echo 'صورة';
                                        else echo 'ملف';
                                        ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo date('Y-m-d H:i', strtotime($m['uploaded_at'])); ?>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="upload_material.php?edit=<?php echo $m['id']; ?>" class="btn" style="background: var(--warning); color: white; padding: 8px 15px; font-size: 14px;">
                                            <i class="fas fa-edit"></i> تعديل
                                        </a>
                                        
                                        <button type="button" class="btn btn-danger delete-btn" style="padding: 8px 15px; font-size: 14px;"
                                                data-id="<?php echo $m['id']; ?>">
                                            <i class="fas fa-trash"></i> حذف
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <div class="no-results" id="noResults">
                    <i class="fas fa-search"></i>
                    <p>لا توجد نتائج تطابق بحثك</p>
                    <p>حاول استخدام كلمات بحث مختلفة</p>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-file"></i>
                    <p>لم تُضف أي مواد بعد</p>
                    <p>استخدم النموذج أعلاه لإضافة أول مادة</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // العناصر
        const fileInput = document.getElementById('fileInput');
        const progressContainer = document.getElementById('uploadProgressContainer');
        const progressBar = document.getElementById('uploadProgress');
        const progressText = document.getElementById('uploadProgressText');
        const uploadForm = document.getElementById('uploadForm');
        const submitBtn = document.getElementById('submitBtn');
        const errorMessage = document.getElementById('errorMessage');
        const successMessage = document.getElementById('successMessage');
        const errorText = document.getElementById('errorText');
        const successText = document.getElementById('successText');
        const studentSearch = document.getElementById('studentSearch');
        const studentsContainer = document.getElementById('studentsContainer');
        const searchMaterials = document.getElementById('searchMaterials');
        const materialsTable = document.getElementById('materialsTable');
        const resultsCount = document.getElementById('resultsCount');
        const noResults = document.getElementById('noResults');

        // ✅ التحقق من وجود fileInput قبل إضافة الحدث
        if (fileInput) {
            fileInput.addEventListener('change', function() {
                if (this.files.length > 0) {
                    progressContainer.style.display = 'block';
                    progressBar.style.width = '0%';
                    progressText.textContent = '0%';
                }
            });
        }

        // رفع الملف
        uploadForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            if (!<?php echo $is_edit ? 'true' : 'false'; ?> && (!fileInput || fileInput.files.length === 0)) {
                showError('يرجى اختيار ملف.');
                return;
            }

            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري المعالجة...';

            try {
                let filePath = '<?php echo $is_edit ? $material['file_path'] : ''; ?>';
                let fileType = '<?php echo $is_edit ? $material['file_type'] : ''; ?>';

                if (!<?php echo $is_edit ? 'true' : 'false'; ?> && fileInput && fileInput.files.length > 0) {
                    const fileFormData = new FormData();
                    fileFormData.append('file', fileInput.files[0]);

                    const xhr = new XMLHttpRequest();
                    xhr.open('POST', 'upload_file.php', true);

                    // ✅ شريط التقدم الحقيقي
                    xhr.upload.onprogress = function(e) {
                        if (e.lengthComputable) {
                            const percentComplete = (e.loaded / e.total) * 100;
                            progressBar.style.width = percentComplete + '%';
                            progressText.textContent = Math.round(percentComplete) + '%';
                        }
                    };

                    const uploadResult = await new Promise((resolve, reject) => {
                        xhr.onload = function() {
                            if (xhr.status === 200) {
                                try {
                                    const result = JSON.parse(xhr.responseText);
                                    if (result.error) {
                                        reject(new Error(result.error));
                                    } else {
                                        resolve(result);
                                    }
                                } catch (e) {
                                    reject(new Error('استجابة غير صالحة من السيرفر: ' + xhr.responseText));
                                }
                            } else {
                                reject(new Error('فشل في الاتصال بالسيرفر. كود الحالة: ' + xhr.status));
                            }
                        };
                        xhr.onerror = function() {
                            reject(new Error('خطأ في الشبكة. تأكد من اتصال الإنترنت.'));
                        };
                        xhr.send(fileFormData);
                    });

                    filePath = uploadResult.file_path;
                    fileType = uploadResult.file_type;
                    
                    formData.set('existing_file', filePath);
                    formData.set('file_type', fileType);
                }

                // حفظ المادة
                const saveResponse = await fetch('upload_material.php', {
                    method: 'POST',
                    body: formData
                });

                const result = await saveResponse.json();

                if (result.error) {
                    throw new Error(result.error);
                }

                showSuccess(result.success);
                
                setTimeout(() => {
                    window.location.href = 'upload_material.php';
                }, 2000);

            } catch (err) {
                showError(err.message);
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fas fa-save"></i> <?php echo $is_edit ? 'تحديث المادة' : 'رفع المادة'; ?>';
            }
        });

        // حذف
        document.querySelectorAll('.delete-btn').forEach(btn => {
            btn.addEventListener('click', async function() {
                if (!confirm('هل أنت متأكد من حذف هذه المادة؟\n\nملاحظة: لا يمكن التراجع.')) {
                    return;
                }

                const materialId = this.dataset.id;

                try {
                    const response = await fetch('upload_material.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            'delete_material': '1',
                            'material_id': materialId
                        })
                    });

                    const result = await response.json();

                    if (result.error) {
                        showError(result.error);
                    } else {
                        showSuccess(result.success);
                        this.closest('tr').remove();
                    }
                } catch (err) {
                    showError('حدث خطأ أثناء الحذف.');
                }
            });
        });

        // ✅ البحث التلقائي عن الطلاب (تم الإصلاح)
        if (studentSearch) {
            let debounceTimer;
            studentSearch.addEventListener('input', function() {
                clearTimeout(debounceTimer);
                const term = this.value.trim();
                
                debounceTimer = setTimeout(async () => {
                    try {
                        const url = new URL(window.location.href);
                        url.searchParams.set('search', term);
                        if (<?php echo $is_edit ? 'true' : 'false'; ?>) {
                            url.searchParams.set('edit', <?php echo $is_edit ? $material['id'] : '0'; ?>);
                        }

                        const response = await fetch(url.toString());
                        const html = await response.text();
                        
                        const parser = new DOMParser();
                        const doc = parser.parseFromString(html, 'text/html');
                        const newContainer = doc.getElementById('studentsContainer');
                        
                        if (newContainer) {
                            studentsContainer.innerHTML = newContainer.innerHTML;
                        }
                    } catch (err) {
                        console.error('Error searching students:', err);
                    }
                }, 500);
            });
        }

        // البحث في المواد
        searchMaterials.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase().trim();
            const rows = materialsTable.querySelectorAll('tbody tr');
            let visibleCount = 0;

            rows.forEach(row => {
                const title = row.dataset.title || '';
                const students = row.dataset.students || '';
                const type = row.dataset.type || '';
                
                if (title.includes(searchTerm) || 
                    students.includes(searchTerm) || 
                    type.includes(searchTerm)) {
                    row.style.display = '';
                    visibleCount++;
                } else {
                    row.style.display = 'none';
                }
            });

            resultsCount.textContent = visibleCount + ' نتيجة';
            
            if (visibleCount === 0 && rows.length > 0) {
                noResults.style.display = 'block';
            } else {
                noResults.style.display = 'none';
            }
        });

        // دوال الرسائل
        function showError(message) {
            errorText.textContent = message;
            errorMessage.style.display = 'flex';
            successMessage.style.display = 'none';
        }

        function showSuccess(message) {
            successText.textContent = message;
            successMessage.style.display = 'flex';
            errorMessage.style.display = 'none';
        }
    });
    </script>
</body>
</html>